#include "files.h"
#include "distribution.h"
#include "asmmodel.h"
#include "options.h"
#include <iostream>
#include <iomanip>
#include <math.h>
#include "fmtout.h"
#include "modelweight.h"
#include "entropy.h"

const Double ALLEGROLOG10 = log(10.0);

ASMmodel::ASMmodel(const string &pt, const string &mo) :
    Model(pt), model(mo) {}

void spacetodot(string &s) {
  for (Uint i = 0; i < s.length(); i++)
    if (s[i] == ' ') s[i] = '.';
}

void ASMmodel::setupfiles(const string &outf, const string &foutf) {
  string desc = distribution->describe().c_str();
  spacetodot(desc);
  setfiles(outf, foutf, model + desc + "." +
           modelweight->describe() + "." + pt, "allele sharing model output");

  string on = outfile.name;
  string::size_type lastslash = on.find_last_of("/");
  string fname;
  if (lastslash == string::npos) fname = "i" + on;
  else fname = (on.substr(0, lastslash + 1) + "i" + on.substr(lastslash + 1));
  ioutfile.setname(fname, fname);
}

void ASMmodel::print() const {
  string famf = (options->famfiles == OB_NOTSET && famfiles ||
                 options->famfiles == OB_ON) ? " " + foutfile.name : "";
  string info = (options->faminfo ? " " + ioutfile.name : "");
  message("MODEL " + pt + " " + model + " " + distribution->describe() + " " +
          modelweight->describe() + " " + outfile.name + famf +
          info);
}

void ASMmodel::initialize() {
  if (options->entropy)
    entropy = Entropy::getentropy(pt);
  
  NEWVEC(Double, dhat, npos());
  NEWVEC(Double, zlr, npos());
  NEWVEC(Double, npl, npos());
  NEWVEC(Double, lod, npos());
  NEWMAT(Double, flod, nfam(), npos());
  NEWMAT(Double, wnpl, nfam(), npos());
  if (Distribution::isnullconstant()) {
    NEWVEC(DoubleVec, nullmean, npos());
    NEWVEC(DoubleVec, nullsd, npos());
    NEWVEC(DoubleVec, weight, npos());
    NEWVEC(Double, nullmean[0], nfam());
    NEWVEC(Double, nullsd[0], nfam());
    NEWVEC(Double, weight[0], nfam());
    for (Uint pos = 1; pos < npos(); pos++) {
      nullmean[pos] = nullmean[0];
      nullsd[pos] = nullsd[0];
      weight[pos] = weight[0];
    }
  }
  else {
    NEWMAT(Double, nullmean, npos(), nfam());
    NEWMAT(Double, nullsd, npos(), nfam());
    NEWMAT(Double, weight, npos(), nfam());
  }  
  NEWMAT(Double, faminfo, nfam(), npos());
  NEWMAT(Double, information, 4, npos());
  // Initialize weights
  modelweight->initialize(distribution->families);
}

void ASMmodel::cleanup() {
  DELETEVEC(dhat);
  DELETEVEC(zlr);
  DELETEVEC(npl);
  DELETEVEC(lod);
  DELETEMAT(flod);
  DELETEMAT(wnpl);
  if (Distribution::isnullconstant()) {
    DELETEVEC(nullmean[0]);
    DELETEVEC(nullsd[0]);
    DELETEVEC(weight[0]);
    DELETEVEC(nullmean);
    DELETEVEC(nullsd);
    DELETEVEC(weight);
  }
  else {
    DELETEMAT(nullmean);
    DELETEMAT(nullsd);
    DELETEMAT(weight);
  }
  DELETEMAT(faminfo);
  DELETEMAT(information);
}

void ASMmodel::assignweights() const {
  for (Uint pos = 0; pos < (Distribution::isnullconstant() ? 1 : npos());
       pos++) {
    Double sumofsquares = 0.0;
    for(Uint fam = 0; fam < nfam(); fam++) {
      weight[pos][fam] = modelweight->assign(getfamid(fam), nullsd[pos][fam]);
      sumofsquares += weight[pos][fam]*weight[pos][fam];
    }
    sumofsquares = sqrt(sumofsquares/Double(nfam()));
    if (sumofsquares != 0.0)
      for(Uint fam = 0; fam < nfam(); fam++) 
        weight[pos][fam] /= sumofsquares;
  }
}

void ASMmodel::compute_final_stats() {
  for(Uint pos = 0; pos < npos(); pos++) {
    Double scale = 0.0;
    for(Uint fam = 0; fam < nfam(); fam++)
      scale += (weight[pos][fam]*weight[pos][fam]);
    scale = sqrt(scale);
    
    npl[pos] = 0.0;
    for(Uint fam=0; fam < nfam(); fam++) 
      npl[pos] += wnpl[fam][pos];
    npl[pos] /= scale;
  }

  Double constant = 2.0*log(10.0);
  for(Uint pos = 0; pos < npos(); pos++) {
    if(dhat[pos] < 0.0) {
      zlr[pos] = -1.0 * sqrt(constant*lod[pos]);
    } else {
      zlr[pos] = sqrt(constant*lod[pos]);
    }
  }
}

void ASMmodel::output() {
  initialize();
  assignweights();
  run();
  compute_fam_info();
  Output::output();
  if (options->faminfo) printfaminfo();
  cleanup();
}

ASMmodel::~ASMmodel() {
  delete modelweight;
}

void ASMmodel::totline(ostream &f, Uint pos) {
  fmtout(f, 10, 4, lod[pos]);
  fmtout(f, 9, 4, dhat[pos]);
  fmtout(f, 9, 4, npl[pos]);
  fmtout(f, 9, 4, zlr[pos]);
  if (donplexactp) fmtout(f, 10, 6, nplexactps[pos], 1, 0.001);
  if (dolodexactp) fmtout(f, 10, 6, lodexactps[pos], 1, 0.001);
  if (model == "exp" || model.substr(0, 4) == "poly")
    fmtout(f, 9, 4, information[2][pos]);
  if (options->entropy) fmtout(f, 9, 4, entropy->info(pos));
  f << "   ";
}

Float sgn(Float x) {return x < 0 ? -1.0 : 1.0;}

void ASMmodel::famline(ostream &f, Uint ifa, Uint pos) {
  Double fl = flod[ifa][pos]*sgn(dhat[pos]);
  Double Z = sqrt(2*ALLEGROLOG10*fabs(fl));
  Z *= sgn(fl);
  f.setf(ios::right, ios::adjustfield);
  fmtout(f, 10, 4, fl);
  fmtout(f, 9, 4, wnpl[ifa][pos]/(weight[pos][ifa] != 0 ?
                                  weight[pos][ifa] : 1.0));
  fmtout(f, 9, 4, Z);
  fmtout(f, 9, 4, faminfo[ifa][pos]);
  if (options->entropy) fmtout(f, 9, 4, entropy->faminfo(ifa, pos));
  f << "   ";
}

void ASMmodel::totheader(ostream &f) {
  f << "     LOD     dhat      NPL      Zlr  ";
  if (donplexactp) f << "nplexactp ";
  if (dolodexactp) f << "lodexactp ";
  if (model == "exp" || model.substr(0, 4) == "poly") f << "   info ";
  if (options->entropy) f << "  entropy";
  f << "   ";
}

void ASMmodel::famheader(ostream &f) {
  f << "     LOD      NPL      Zlr    ";
  f << " info ";
  if (options->entropy) f << "  entropy";
  f << "   ";
}
