#ifndef BASIC
#define BASIC
#include "config.h"

#include <new>
#include <string>
#include <vector>

#undef HAVE_ZLIB

#ifdef HAVE_HASH_MAP
#include <hash_map>
#elif HAVE_EXT_HASH_MAP
using namespace __gnu_cxx;
#include <ext/hash_map>
#else
#error "No hash_map found!"
#endif // HASH_MAP

#ifdef HAVE_IEEEFP_H
#include <ieeefp.h>
#endif // HAVE_IEEEFP_H

#include <math.h>
#include <ctype.h>

using namespace std;

////////////////////////////////////////////////////////////////////////////
// Primary typedefs
typedef double            Float;
typedef double            Double;
typedef int               Allele;
typedef int               Sex;
typedef unsigned long int IV;
typedef unsigned long int Uint;

////////////////////////////////////////////////////////////////////////////
// Constanstants and enums
const int MALE = 0;
const int FEMALE = 1;
const Allele ALLELEUNKNOWN = 0;
const Uint MAXBITS = 32;

enum Diseasestatus{UNKNOWN, UNAFFECTED, AFFECTED};
enum Unit {CENTIMORGAN, RECOMBINATION, DEFAULT};

extern unsigned char *wt; // How many bits are 1 in a vector

const unsigned int POW2[32] = {1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024,
                               2048, 4096, 8192, 16384, 32768, 65536, 131072,
                               262144, 524288, 1048576, 2097152, 4194304, 
                               8388608, 16777216, 33554432, 67108864, 
                               134217728, 268435456, 536870912, 
                               1073741824};

////////////////////////////////////////////////////////////////////////////
// Assert functions
#ifdef DEBUG_ASSERT
#include <assert.h>
#define ASSERT_FALSE assert(false)
#else
#define ASSERT_FALSE
#endif // DEBUG_ASSERT

void message(const string &);
string operator+(const string& s, int n);

#define assertcond(cond,msg) { \
  if (!(cond)) { \
    message(string("ERROR ") + (msg)); \
    ASSERT_FALSE; \
    exit(1); \
  } \
}

#ifdef DECODE
#define assertinternal(cond) { \
  if (!(cond)) { \
    message(string("ERROR internal error in ") + __FILE__ + " at line " + __LINE__ + "!"); \
    ASSERT_FALSE; \
    exit(1); \
  } \
}
#else
#define assertinternal(cond) { \
  if (!(cond)) { \
    message("ERROR internal error!!"); \
    exit(1); \
  } \
}
#endif // DECODE

///////////////////////////////////////////////////////////////////////////
// Hashtable typedefs
template<class T>
struct compareconstchar {
  bool operator()(const T& a, const T& b) const
    {return strcmp(a, b) == 0;};
};

struct stringhash {
private:
  hash<const char*> hasher;
  
public:
  size_t operator()(const string &a) const
    {return hasher(a.c_str());}
};

struct comparestrings {
  bool operator()(const string &a, const string &b) const
    {return a == b;}
};

typedef hash_map<string, Double, stringhash, comparestrings > String2Double;
typedef hash_map<string, bool, stringhash, comparestrings > String2Bool;
typedef hash_map<string, Uint, stringhash, comparestrings > String2Uint;

///////////////////////////////////////////////////////////////////////////
// char *indexed table
template<class T>
class Key2Class {
public:
  typedef hash_map<const char *, T*, hash<const char *>,
    compareconstchar<const char *> > container;
  container rep;
  T *get(const char *key, bool insert = false) {
    typename container::iterator it = rep.find(key);
    T *res = (it == rep.end()) ? 0 : it->second;
    if (res == 0 && insert) {
      res = new T(key);
      char *tc = new char[strlen(key) + 1];
      strcpy(tc, key);
      rep.insert(container::value_type(tc, res));
    }
    return res;
  }
};

///////////////////////////////////////////////////////////////////////////
// Class for checking arrays 
#ifdef CHECKMEMORY
template <class T, class Tp, class Tpp>
class MemCheckedVec {
protected:
  static vector<T *> allocated;
  
  int data;
  Uint size;
  enum Status{STATUS_UNINITIALIZED, STATUS_NULL, STATUS_VALID};
  Status status;
  int offset;
public:
  MemCheckedVec(int i = 0) : data(-1), size(0),
    status(i == 0 ? STATUS_NULL : STATUS_UNINITIALIZED), offset(0)
    {assertinternal(i == 0);}
  MemCheckedVec(const MemCheckedVec<T, Tp, Tpp> &v) : data(v.data), size(v.size),
    status(v.status), offset(v.offset) {}
  MemCheckedVec(void *x) : data(-1), size(-1), status(STATUS_VALID), offset(0) {
    for (Uint i = 0; i < allocated.size() && data == -1; i++)
      if ((T *)x == allocated[i]) data = i;
    if (data == -1) {
      allocated.push_back((T *)x);
      data = allocated.size() - 1;
    }
  }

  void allocate(Uint n) {
    size = n;
    allocated.push_back(new T[size]);
    data = allocated.size() - 1;
    status = STATUS_VALID;
  }
  void deallocate() {
    if (status != STATUS_NULL) {
      assertcond(status != STATUS_UNINITIALIZED, "Deleting an invalid vector");
      assertinternal(data >= 0);
      delete [] allocated[data];
      allocated[data] = 0;
      status = STATUS_UNINITIALIZED;
      data = -1;
    }
  }
  T operator [](int idx) const {
    assertcond(status == STATUS_VALID,
               string("Trying to use an invalid vector"));
    assertcond(data >= 0 && allocated[data] != 0,
               string("Invalid vector read"));
    idx = idx + offset;
    assertcond(idx >= 0 && Uint(idx) < size,
               string("Memory reference :") + idx + " out of bounds");
    return allocated[data][idx];
  }
  T &operator [](int idx) {
    assertcond(status == STATUS_VALID,
               string("Trying to use an invalid vector"));
    assertcond(data >= 0 && allocated[data] != 0,
               string("Invalid vector read"));
    idx = idx + offset;
    assertcond(idx >= 0 && Uint(idx) < size,
               string("Memory reference :") + idx + " out of bounds");
    return allocated[data][idx];
  }  
  const MemCheckedVec &operator =(const MemCheckedVec &v) {
    data = v.data;
    size = v.size;
    status = v.status;
    offset = v.offset;
    return *this;
  }
  const MemCheckedVec &operator =(int i) {
    assertinternal(i == 0);
    data = -1;
    size = 0;
    status = STATUS_NULL;
    return *this;
  }
  bool operator ==(int i) const {return status == STATUS_NULL;}
  bool operator !=(int i) const {return status != STATUS_NULL;}
  bool operator ==(const MemCheckedVec &v) const {
    return (v.status == STATUS_NULL && status == STATUS_NULL ||
            data >= 0 && v.data == data && offset == v.offset);
  }
  friend MemCheckedVec operator +(const MemCheckedVec &v, int i) {
    MemCheckedVec result = v;
    result.offset += i;
    return result;
  }
  Uint get_size() const {return size;}
  Tp get_vec_data() {
    assertinternal(offset >= 0 && Uint(offset) < size);
    assertinternal(status == STATUS_VALID);
    return allocated[data];
  }
  Tpp get_mat_data() {
    Tpp x = new Tp[size];
    for (Uint i = 0; i < size; i++) x[i] = ((*this)[i]).get_vec_data();
    return x;
  }
};

#define Vec(type) MemCheckedVec<type, type *, type **>
#define Mat(type) MemCheckedVec<Vec(type), type *, type **>
#define VEC_GET_DATA .get_vec_data()
#define MAT_GET_DATA .get_mat_data()

#define NEWVEC(type,x,n) (x).allocate(n)
#define DELETEVEC(x) (x).deallocate();

#else // CHECKMEMORY

#define Vec(type) type *
#define Mat(type) type **
#define VEC_GET_DATA
#define MAT_GET_DATA

#define NEWVEC(type,x,n) (x) = new (type)[n]
#define DELETEVEC(x) delete [] (x)

#endif // CHECKMEMORY

///////////////////////////////////////////////////////////////////////////
// Vector and matrix types
typedef Vec(Double) DoubleVec;
typedef Vec(Float) FloatVec;
typedef Vec(int) IntVec;
typedef Vec(bool) BoolVec;
typedef Vec(Uint) UintVec;
typedef Vec(IV) IVVec;
typedef Vec(string) StringVec;

typedef Mat(Double) DoubleMat;
typedef Mat(Float) FloatMat;
typedef Mat(int) IntMat;
typedef Mat(Uint) UintMat;
typedef Mat(IV) IVMat;

#define NEWMAT(type,x,n,m) (x) = newmatrix<type>((n), (m))
#define DELETEMAT(x) deletematrix(x)

typedef vector<string> Stringvector;
typedef vector<Double> Doublevector;
typedef vector<Uint> Uintvector;
typedef vector<bool> Boolvector;

////////////////////////////////////////////////////////////////////////////
// Memory management

void *operator new(size_t size) throw(bad_alloc);
void *operator new[](size_t size) throw(bad_alloc);

template<class T>
Mat(T) newmatrix(Uint m, Uint n) {
  Mat(T) x;
  if (m == 0) x = 0;
  else {
    NEWVEC(Vec(T), x, m);
    if (n == 0) for (Uint i = 0; i < m; i++) x[i] = 0;
    else {
      NEWVEC(T, x[0], n*m);
      for (Uint i = 1; i < m; i++)
        x[i] = x[i - 1] + n;
    }
  }
  return x;
}

template<class T>
Mat(T) newsymmetricmatrix(Uint n) {
  Mat(T) x;
  if (n == 0) x = 0;
  else {
    NEWVEC(Vec(T), x, n);
    NEWVEC(T, x[0], (n*(n + 1))/2);
    for (Uint i = 1; i < n; i++)
      x[i] = x[i - 1] + i;
  }
  return x;
}

template<class T>
void deletematrix(T &x) {
  if (x != 0) {
    DELETEVEC(x[0]);
    DELETEVEC(x);
    x = 0;
  }
}

///////////////////////////////////////////////////////////////////////////
// Very commonly used classes
class Family;
class Person;
class Map;

#endif // _BASIC

/*
Notational conventions
* In comments, TeX style subscripts are a_i and superscripts a^k
* In general underscores are not used in identifiers
* Class names and variable names are in singular
* Typedefs and class names have initial capital, with a few exceptions
  (e.g. Uint, Float, Double)
* Variable and function names are all lowercase with few exceptions,
  (e.g. findMgam, N)
* Constants (including enum constants) are all capitals
* Sometimes the class name is spelled out in full and an abbreviation is used 
  for an instantiation. In other cases it is just the initial capital in the 
  class name that distinguishes (Sex and sex). The third possibility is to 
  use different words for class and variable (Person and member)
*/
