#include "files.h"
#include "options.h"
#include "asmmodel.h"
#include "asmpoly.h"
#include "asmexp.h"
#include "control.h"
#include "simulation.h"
#include "matrix.h"
#include "outputld.h"
#include "vecutil.h"
#include "probability.h"
#include "utils.h"
#include "calcscore.h"
#include "fmtout.h"
#include "distribution.h"
#include "haldane.h"
#include "calcviterbi.h"
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include "family.h"

Control *control = 0;

const string plurals(Uint x) {return x > 1 ? "s" : "";}

Control::Control() :
    first(0), firstorig(0), maxnumiv(0) {
  assertinternal(options != 0);
  if (options->traitvaluefile.assigned()) {
    options->traitvaluefile.open();
    char buf[1000];
    Uint linenum = 0;
    while (!options->traitvaluefile.eof()) {
      linenum++;
      options->traitvaluefile >> buf;
      Double traitval;
      options->traitvaluefile >> traitval;
      assertcond(!options->traitvaluefile.fail() && finite(traitval),
                 string("Illegal trait value in trait value file at line ") +
                 linenum);
      string2traitvalue[string(buf)] = traitval;
      options->traitvaluefile >> ws;
    }
    options->traitvaluefile.close();
  }
  if (options->prefile.size() > 0 && options->datfile.size() > 0) {
    // Read dat-file and pre-file
    inputld();
  }
  else if (options->prefile.size() > 0 && options->genotypefile.assigned() &&
           options->mapfile.assigned() && options->freqfile.assigned()) {
    inputdecode();
  }
  else {
    // Read family names from the first nullprob file that is assigned    
    assertinternal(false);
  }
}

Control::~Control() {
  deletelist(firstorig);
  delete [] wt;
}

Float Control::countvecsneeded() {
  Double count = 0.0;
  count += Calcscore::countvecsneeded(first, maxnumiv);
  return count;
}


void Control::probs() {
  if (options->chromolist == 0) {
    // The usual; use linkage input or probfiles
    analyzechromosome();
  }
  else {
    // New input format
    if (options->chromolist->all) {
      for (String2Chromosome::container::iterator ch =
             string2chromosome.rep.begin();
           ch != string2chromosome.rep.end(); ch++) {
        string id = ch->first;
        initializechromosome(id);
        analyzechromosome();
      }
    }
    else {
      for (Chromolist *cl = options->chromolist; cl != 0; cl = cl->next) {
        initializechromosome(cl->id);
        analyzechromosome();
      }
    }
  }
  Distribution::cleanup();
}

void Control::analyzechromosome() {
  Distribution::setnullconstant(!options->pseudoautosomal);
  setupfamilies();
  Output::map = &map;
  if (options->chromolist != 0) {
    writepre("pre", first, &map);
    writedat("dat", &map);
  }
  
  Uint nq, nlq, nlqhat;
  nq = nlq = nlqhat = 0;
  FloatVec lqbuf =  0;
  FloatVec lqhatbuf = 0;
  FloatVec qbuf = 0;
  FloatVec rqhat = 0;
  Floatmatrix *q = 0, *lq = 0, *lqhat = 0;
  DoubleVec simbuf = 0;
  q = lq = lqhat = 0;
//  if (!options->readprobfiles || !options->skipfirstphase) initscore();
  findnums();
  Double count = countvecsneeded() + 1.0;
  options->swap = false;
  if (options->calcmpt || options->calcdpt ||
      options->dohaplo || options->doforcedrecomb) {
    string mesg;
    bool recalcallowed = true;//options->markerdistzero || options->chromolist == 0;
    Double allfull = 0;   // Number of vecs needed to be stored in memory
    Double recalc1full = 0; // Number of vecs with some recalculation
    Double recalc2full = 0; // Number of vecs with more recalculation
    Double allone = count;
    Double recalc1one = count;
    Double recalc2one = count;
    Double swapone = count;
    if (options->calcmpt || options->calcdpt) {
      allfull = 3.0;
      recalc1full = 2.0;
      recalc2full = 1.0;
      recalc1one += 1;
      recalc2one += 2;
      swapone = 3;
      if (options->dohaplo && options->doforcedrecomb) {
        allfull += .5;
        recalc1full += .5;
        recalc2full += .5;
        swapone += 1;
      }
      else if (options->dohaplo || options->doforcedrecomb) swapone += .5;
    }
    else if (options->dohaplo && options->doforcedrecomb) {
      recalc2full = recalc1full = allfull = 1;
      swapone += 1;
    }
    else if (options->dohaplo || options->doforcedrecomb) {
      recalc2full = recalc1full = allfull = .5;
      swapone = .5;
    }
    Double vecsneeded = 3.0;
    if (Double(options->maxmem) >=
        Double(maxnumiv)*Double(allfull*map.num + allone)*sizeof(Float)) {
      // Plenty of memory
      nq = nlq = nlqhat = maxnumiv*map.num;
      vecsneeded = allfull*map.num + allone;
      mesg = "Keeping all calculations in memory";
    }
    else if (Double(options->maxmem) >=
             Double(maxnumiv)*Double(recalc1full*map.num +
                                     recalc1one)*sizeof(Float)) {
      // Need to recalculate lq from lqhat
      nq = nlqhat = maxnumiv*map.num;
      nlq = maxnumiv;
      vecsneeded = recalc1full*map.num + recalc1one;
      mesg = "Recalculating lq, keeping other in memory";
    }
    else if (recalcallowed && Double(options->maxmem) >=
             Double(maxnumiv)*Double(recalc2full*map.num +
                                     recalc2one)*sizeof(Float)) {
      // Need to recalculate lq from lqhat and q from scratch
      nlqhat = maxnumiv*map.num;
      nq = nlq = maxnumiv;
      vecsneeded = recalc2full*map.num + recalc2one;
      mesg = "Recalculating lq and q, keeping other in memory";
    }
    else if (recalcallowed && Double(options->maxmem) >=
             Double(maxnumiv)*(swapone)*sizeof(Float)) {
      // Need to recalculate lq and q, and write lqhat to disk
      nq = nlq = nlqhat = maxnumiv;
      vecsneeded = swapone;
      mesg = "Recalculating lq and q, swapping lqhat to disk";
      options->swap = true;
    }
    else
      fatal(string("Allegro does not have permission to allocate enough ") +
            "memory.\nRequires at least " +
            Double(maxnumiv)*(vecsneeded)*sizeof(Float) +
            " bytes, but MAXMEMORY is set to " + options->maxmem + "\n");
    message(mesg + " (~" + ceil(Double(maxnumiv)*vecsneeded*
                                sizeof(Float)/(1024.0*1024.0)) + "Mb)");
    if (options->estimatememory)
      return;
    if (options->swap) {
      message("SWAPDIRNAME " +  options->swapdirname);
      assertcond(mkdir(options->swapdirname.c_str(), 493) == 0,
                 "Unable to creat swap directory or it already exists!");
    }
    if (options->calcmpt) {
      NEWVEC(Float, lqhatbuf, nlqhat);
      Calcviterbi::registerbuf(IVVec(lqhatbuf + nlqhat/2), nlqhat);
      NEWVEC(Float, lqbuf, nlq);
      NEWVEC(Float, rqhat, maxnumiv);
      lq = new Floatmatrix(lqbuf, nlq, Floatmatrix::RECALCLQ);
      lqhat = new Floatmatrix(lqhatbuf, nlqhat, options->swapdirname, "lqhat.");
    }
  }
  else {
    if (Float(options->maxmem) <
        Float(maxnumiv)*(1 + count)*sizeof(Float))
      fatal(string("Allegro does not have permission to allocate enough ") +
             "memory.\nRequires at least " + maxnumiv*(1 + count)*sizeof(Float) +
             " bytes, but maxmem is set to " + options->maxmem + "\n");
    nq = maxnumiv;
    if (options->pseudoautosomal)
      NEWVEC(Float, rqhat, maxnumiv);
  }
  message("");
  NEWVEC(Float, qbuf, nq);

  if (options->rfile) options->rfile.close();
  if (options->pfile) options->pfile.close();    

  q = new Floatmatrix(qbuf, nq, Floatmatrix::RECALCQ);
  // Initialize model calculations (do not change this order)
  Calcscore::reset(first, maxnumiv);
  Distribution::reset(map);
  // Initialize simulations
  if (options->simulation) {
    NEWVEC(Double, simbuf, maxnumiv);
    options->simulation->setbuf(simbuf);
  }
  // Open file for uninformative markers
  if (options->uninformative.assigned()) options->uninformative.open();
  // Open file for inconsistent markers
  if (options->inconsistent.assigned()) options->inconsistent.open();
  // Open file for likelihood function
  if (options->likelihoodfile.assigned()) options->likelihoodfile.open();

  if (options->rfile.assigned()) options->rfile.open();
  if (options->pfile.assigned()) options->pfile.open();

  for (Family *fam = first; fam != 0; fam = fam->next) {
    message(string("Processing family ") + fam->id + " (" +
            fam->numbits + " bit" + plurals(fam->numbits) + ")");
    Probability prob(q, lq, lqhat, rqhat, fam, &map);
    // Perform simulations
   if (options->simulation != 0) {
      options->simulation->simulate(fam, map);
      continue;
    }

    q->nextfam(map.num*fam->numiv, fam->numiv);
    q->probability = &prob;
    if (options->calcmpt) {
      lqhat->nextfam(map.num*fam->numiv, fam->numiv);
      lq->nextfam(map.num*fam->numiv, fam->numiv);
      lq->probability = &prob;
    }
    // Calculate Likelihood and scorefunctions
    Distribution::nextfam(prob, fam, qbuf);
    // Calculate single locus probabilities and fromleft (i.e. find l and q)
    if (fromleft(fam, prob, q, lq, lqhat, rqhat)) {
      // 'Fromright' and multipoint analysis
      if (options->calcmpt || options->dohaplo || options->doforcedrecomb)
        fromright(prob, q, lq, fam);
      prob.deletefoundergraph();
      if (options->writeprobfiles) {
        Distribution::writeprobfiles();
        Distribution::writenullfiles();
      }
    }
    else {
      // Family contains no informative markers so we must pass uniform to
      // all distributions that are still interested
      copyval(qbuf, 1.0/Double(fam->numiv), fam->numiv);
      if (options->calcmpt)
        for (Uint pos = 0; pos < map.numposition; pos++)
          Distribution::set(qbuf, pos, "mpt");
      if (options->calcdpt)
        for (Uint gam = 0; gam < map.num; gam++)
          Distribution::set(qbuf, gam, "dpt");
      if (options->dohaplo)
        for (Uint gam = 0; gam < map.num; gam++)
          Distribution::set(qbuf, gam, "hpt");
      if (options->doforcedrecomb)
        for (Uint gam = 0; gam < map.num; gam++)
          Distribution::set(qbuf, gam, "fpt");
      if (options->montecarlo) {
        prob.rightmarker[0] = Probability::NORIGHTMARKER;
        prob.leftmarker[0] = 0;
        Distribution::set(qbuf, 0, "cpt");
        Distribution::set(qbuf, 0, "cpt");
      }
    }
  }
  
  message("");
  message("All families processed");
  message("");

  options->uninformative.close();
  options->inconsistent.close();
  options->likelihoodfile.close();

  DELETEVEC(simbuf);
  delete q;
  DELETEVEC(qbuf);
  if (options->calcmpt) {
    delete lq;
    delete lqhat;
    DELETEVEC(lqbuf);
    DELETEVEC(lqhatbuf);
  }
  DELETEVEC(rqhat);
  
  // Open file for delta bounds
  if (options->deltaboundfile.assigned()) options->deltaboundfile.open();

  Model::modeloutput();

  options->deltaboundfile.close();

  if (options->swap)
    rmdir(options->swapdirname.c_str());
}

void Control::readprobfiles() {
//    message("Reading distributions from nullprob and prob files");
//    for (Uint scoret = 0; scoret < nscore; scoret++) {
//      bool nullread = false;
//      for (Uint pt = 0; pt < npt; pt++) {
//        if (dist[scoret][pt]) {
//          if (!nullread) {
//            dist[scoret][pt]->readnull(options->nullfilein[scoret]);
//            nullread = true;
//          }
//          dist[scoret][pt]->readprob(options->probfilein[scoret][pt]);
//        }
//      }
//    }
//    message("");
//    message("All families processed");
//    message("");
}

bool Control::fromleft(Family *f, Probability &prob, Floatmatrix *q,
                       Floatmatrix *lq, Floatmatrix *lqhat, FloatVec tmpq) {
  if (!options->calcspt) return true;
  int lastinformative = options->calcmpt ? Probability::NOLEFTMARKER : 0;
  Uint curthetaidx = 0;
  if (options->sexspecific) {
    prob.theta = map.theta[1];
    prob.theta_female = map.theta[2];
  }
  else prob.theta = map.theta[0];
  Float L = 0.0; // The likelihood function value for this family
  if (options->lfile.assigned()) options->lfile.open();
  for (Uint gam = 0; gam < map.num; gam++) {
    FloatVec curq = q->getrow(gam, false);
    L += log(prob.fromhere(gam, curq));
    if (gam == 0) L -= log(double(f->numiv));
    if (!prob.informative[gam] &&
        lastinformative != Probability::NOLEFTMARKER &&
        gam < map.num - 1) {
      prob.theta[curthetaidx] = addrec(prob.theta[curthetaidx],
                                       prob.theta[gam]);
      if (options->sexspecific)
        prob.theta_female[curthetaidx] = addrec(prob.theta_female[curthetaidx],
                                                prob.theta_female[gam]);
    }
    if (options->pseudoautosomal) {
      f->pseudonull(tmpq);
      fft(tmpq, tmpq, f->numiv, f->numbits);
      prob.Ttrans(tmpq, tmpq, recombfraccent(map.position[0][gam]),
                  recombfraccent(map.position[1][gam]),
                  recombfraccent(map.position[2][gam]));
      fft(tmpq, tmpq, f->numiv, f->numbits);
      elemprod(tmpq, curq, tmpq, f->numiv);
      assertinternal(normal<Float>(tmpq, f->numiv) != 0);
      Distribution::set(tmpq, gam, "spt");
    }
    else
      Distribution::set(curq, gam, "spt");
    if (options->calcmpt && prob.informative[gam]) {
      if (lastinformative == Probability::NOLEFTMARKER) {
        // This is the leftmost informative marker
        if (options->pseudoautosomal) {
          f->pseudonull(tmpq);
          fft(tmpq, tmpq, f->numiv, f->numbits);
          prob.Ttrans(tmpq, tmpq, recombfraccent(map.position[0][gam]),
                      recombfraccent(map.position[1][gam]),
                      recombfraccent(map.position[2][gam]));
          fft(tmpq, tmpq, f->numiv, f->numbits);
          elemprod(lq->getrow(gam, false), q->getrow(gam, true), tmpq,
                   f->numiv);
          fft(lqhat->getrow(gam, false), lq->getrow(gam, true), f->numiv,
              f->numbits);
        }
        else {
          copyvec(lq->getrow(gam, false), q->getrow(gam, true), f->numiv);
          fft(lqhat->getrow(gam, false), lq->getrow(gam, true), f->numiv,
              f->numbits);
        }
      }
      else { // There are informative markers left of this one
        prob.findlq(lqhat->getrow(lastinformative, true),
                    lqhat->getrow(gam, false), lq->getrow(gam, false),
                    q->getrow(gam, true), lastinformative, gam, L);
        L -= log(double(f->numiv)); // fft normalizing constant
        L += f->numfc*log(1 - prob.theta[lastinformative]);
      }
      lqhat->store(gam);
    }
    else if (gam != 0)
      L -= log(double(f->numiv));
    if (prob.informative[gam]) {
      lastinformative = gam;
      curthetaidx = gam;
    }
  }
  // Calculate the likelihood function
  if (options->likelihoodfile.assigned()) {
    if (f->numbits > 0 && lastinformative != Probability::NOLEFTMARKER)
      L += log(sum<Float>(lq->getrow(lastinformative, true), f->numiv));
    options->likelihoodfile << f->id << "\t";
    fmtout(options->likelihoodfile, 24, 20, L, 1, 9.9e300);
    options->likelihoodfile << endl;
  }
//   if (options->markerdistzero && options->calcmpt) {
//     // Check if markers at the same locations are consistent
//     Float lastinformativepos = -1;
//     int lastinformativeindex = -1;
//     bool firstreppos = true;
//     for (Uint gam = 0; gam < map.num; gam++) {
//       if (prob.informative[gam]) {
//         if (lastinformativepos == map.markerpos[0][gam]) {
//           if (firstreppos) {
//             copyvec(tmpq, q->getrow(lastinformativeindex, true), f->numiv);
//             firstreppos = false;
//           }
//           elemprod(tmpq, tmpq, q->getrow(gam, true), f->numiv);
//           if (sum<Float>(tmpq, f->numiv) == 0) {
//             warning(string("Markers at ") + map.markerpos[0][gam] +
//                     " have inconsistent genotypes");
//             firstreppos = true;
//             for (Uint gm = lastinformativeindex; gm < map.num; gm++)
//               if (prob.informative[gm] &&
//                   lastinformativepos == map.markerpos[0][gm]) {
//                 prob.markuninformative(gm);
//                 copyval(q->getrow(gm, false), 1.0/Float(f->numiv), f->numiv);
//               }
//           }
//         } else {
//           if (lastinformativepos > -1 && !firstreppos) {
//             for (Uint gm = lastinformativeindex; gm < map.num; gm++)
//               if (prob.informative[gm] &&
//                   lastinformativepos == map.markerpos[0][gm])
//                 copyvec(q->getrow(gm, false), tmpq, f->numiv);
//           }
//           lastinformativeindex = gam;
//           lastinformativepos = map.markerpos[0][gam];
//           firstreppos = true;
//         }
//       }
//     }
//   }
  if (options->qfile) options->qfile.close();
  if (options->lfile) options->lfile.close();
  if (lastinformative == Probability::NOLEFTMARKER) {
    // skip families with no informative markers
    warning(f->id + " has no informative markers");
    Distribution::curfamuninformative();
    return false;
  }
  return true;
}

void Control::fromright(Probability &prob, Floatmatrix *q, Floatmatrix *lq,
                        Family *fam) {
  prob.ignoreuninformative();
  FloatVec p = 0;
  if (options->calcmpt || options->calcdpt) {
    p = lq->getrow(map.num - 1, false);
    lq->getrow(0, false);
  }
  for (int pos = map.numposition - 1; pos >= 0; pos--) {
    // Monte carlo simulation
    if (options->montecarlo && !map.inbetween[pos] &&
        prob.informative[map.leftmarker[pos]]) {
      Uint gam = prob.leftmarker[pos];
      Distribution::set(lq->getrow(gam, true), pos, "cpt");
      Distribution::set(q->getrow(gam, true), pos, "cpt");
    }
    // Multipoint caluclations (and dpt)
    if (options->calcmpt || options->calcdpt && map.inbetween[pos]) {
      prob.findp(pos, p); // from right (find r and p)
      Distribution::set(p, pos, "mpt");
    }
    // Haplotype and forced recombination calculations
    if ((options->dohaplo || options->doforcedrecomb) &&
        !map.inbetween[pos]) {
      Uint gam = map.leftmarker[pos];
      FloatVec qp = q->getrow(gam, true);
      if (options->dohaplo)
        Distribution::set(qp, gam, "hpt");
      if (options->doforcedrecomb) {
        for (IV v = 0; v < fam->numiv; v++) if (qp[v] > 0) qp[v] = 1.0;
        Distribution::set(qp, gam, "fpt");
      }
    }
  }
}

unsigned char *wt = 0; // How many bits are 1 in a vector

void Control::findnums() {
  for (Family *fam = first; fam != 0; fam = fam->next) fam->countbits();

  maxnumiv = 0;
  numfam = 0;
  maxbits = -1;
  string largestfam = "";
  options->maxfamidlen = 7;
  options->maxperidlen = 0;
  for (Family *f = first; f != 0; f = f->next) {
    numfam++;
    for (Person *p = f->first; p != 0; p = p->next)
      if (p->id.length() > options->maxperidlen)
        options->maxperidlen = p->id.length();
    if (f->id.length() > options->maxfamidlen)
      options->maxfamidlen = f->id.length();
    if (f->numiv > maxnumiv) maxnumiv = f->numiv;
    int nb = f->numbits;
    if (nb > maxbits) {
      maxbits = nb;
      largestfam = f->id;
    }
  }
  assertcond(Float(maxnumiv)*sizeof(unsigned char) < Float(options->maxmem),
             string("Way to many bits(") + maxbits + ") for current MAXMEMORY");
  static IV wtlength;
  if (maxnumiv == 0) {
    warning("Nothing to be done");
    exit(0);
  }
  if (wt == 0 || maxnumiv != wtlength && !options->estimatememory) {
    delete [] wt;
    wt = new unsigned char[maxnumiv];
    wtlength = maxnumiv;
    Uint k = 1;
    wt[0] = 0;
    while (k < maxnumiv) {
      for (Uint j = 0; j < k; j++) wt[k+j] = wt[j] + 1;
      k *= 2;
    }
  }
  
  if (numfam == 1)
    message(string("Analysing 1 family (") + largestfam + ", " + maxbits
            + " bit" + plurals(maxbits) + ")");
  else
    message(string("Analysing ") + numfam + " families, the largest is " +
            maxbits + " bit" + plurals(maxbits) + "(" + largestfam + ")");
}

void Control::setupfamilies() {
  first = 0;
  Family *lf = 0;
  for (Family *fam = firstorig; fam != 0; fam = fam->next) {
    if (fam->first == 0) {
      if (lf != 0) lf->next = fam->next;
    }
    else {
//        if (options->chromolist == 0) {
//          string nextid = (fam->next == 0 ? "" : fam->next->id);
//          fam->setup();

//          if (first == 0) lf = first = fam;
//          if (nextid != "")
//            while (lf->next->id != nextid) {
//              if (lf->next->first == 0) lf->next = lf->next->next;
//              else lf = lf->next;
//            }
//        }
//        else {
      if (first == 0) lf = first = new Family(*fam);
      else lf = lf->next = new Family(*fam);
      lf->setup();
      while (lf->next != 0) {
        if (lf->next->first == 0) lf->next = lf->next->next;
        else lf = lf->next;
      }
//        }
    }
  }
  // Remove empty families
  while (first != 0 && first->first == 0) first = first->next;
  lf = first;
  if (lf != 0) {
    while (lf->next != 0)
      if (lf->next->first == 0) lf->next = lf->next->next;
      else lf = lf->next;
  }
  
//    // Set liability classes
//    for (Family *fam = first; fam != 0; fam = fam->next) {
//      for (Person *p = fam->first; p != 0; p = p->next)
//        p->liability = (options->sexlinked ? 1 - p->sex : 0);
//    }
}
