#include "files.h"
#include "family.h"
#include "warning.h"
#include "options.h"
#include "kinship.h"
#include "vecutil.h"
#include "utils.h"

///////////////////////////////////////////////////////////////////////////
// Person
void Person::init(const string& pid, Sex sx) {
  gen[0] = gen[1] = 0;
  genotyp = 0;
  id = pid;
  sex = sx;
  origdstat = dstat = UNKNOWN;
  liability = 0;
  father = mother = next = 0;
  children = 0;
  nod[0] = nod[1] = 0;
  visitedup = visiteddown = false;
  processed = false;
  traitvalue = NOTRAITVALUE;
  mask = lastbit = 0;
  patmask = matmask = 0;
  nmkr = nmrk = 0;
}

bool Person::ishodgeinformativeup(Uint ngen) {
  if (ngen > 1 && dstat == AFFECTED && genotyped) return true;
  else if (!visitedup && !visiteddown) {
    visitedup = true;
    if (!founder()) {
      if ((!options->sexlinked || sex == FEMALE) &&
          father->ishodgeinformativeup(ngen + 1)) return true;
      if (mother->ishodgeinformativeup(ngen + 1)) return true;
    }
    for (Plist *c = children; c != 0; c = c->next)
      if (!c->p->visitedup && (!options->sexlinked || sex == FEMALE ||
                               c->p->sex == FEMALE))
        if (c->p->ishodgeinformativedown(ngen + 1)) return true;
  }
  return false;
}

bool Person::ishodgeinformativedown(Uint ngen) {
  if (visiteddown) return false;
  else if (ngen > 1 && dstat == AFFECTED && genotyped) return true;
  else {
    visiteddown = true;
    for (Plist *c = children; c != 0; c = c->next)
      if ((!options->sexlinked || sex == FEMALE || c->p->sex == FEMALE) &&
          c->p->ishodgeinformativedown(ngen + 1)) return true;
    return false;
  }
}

Person::Person(const string& pid, Sex sx, Diseasestatus ds, Uint liab,
               Double trval) {
  init(pid, sx);
  origdstat = dstat = ds;
  liability = liab;
  genotyped = false;
  fc = 0;
  traitvalue = trval;
}

Person::Person(const Person &p) {
  init(p.id, p.sex);
  origdstat = p.origdstat;
  dstat = p.dstat;
  traitvalue = p.traitvalue;
  liability = p.liability;
  fc = 0;
  if (p.genotyped) setgenotypes(p.gen[0], p.gen[1], p.nmkr);
  genotyped = p.genotyped;
}

void Person::allocgenotypes(Uint nm) {
  nmkr = nm;
  DELETEVEC(gen[0]);
  DELETEVEC(gen[1]);
  DELETEVEC(genotyp);
  NEWVEC(Allele, gen[0], nmkr);
  copyval(gen[0], ALLELEUNKNOWN, nmkr);
  NEWVEC(Allele, gen[1], nmkr);
  copyval(gen[1], ALLELEUNKNOWN, nmkr);
  NEWVEC(bool, genotyp, nmkr);
  copyval(genotyp, false, nmkr);
}

void Person::setgenotypes(IntVec g0, IntVec g1, Uint nm) {
  allocgenotypes(nm);
  bool unknown = g0 == 0;
  genotyped = false;
  for (Uint i = 0; i < nmkr; i++) {
    gen[0][i] = unknown ? 0 : g0[i];
    gen[1][i] = unknown ? 0 : g1[i];
    genotyp[i] = (gen[0][i] != ALLELEUNKNOWN && gen[1][i] != ALLELEUNKNOWN);
  }
}

Person::~Person() {
  DELETEVEC(gen[0]);
  DELETEVEC(gen[1]);
  DELETEVEC(genotyp);
  delete children;
}

Uint Person::visit() { // used by "bool connected()"
  if (visitedup) return 0;
  Uint n = 1;
  visitedup = true;
  for (Plist *c = children; c != 0; c = c->next) n += c->p->visit();
  if (!founder()) {
    n += mother->visit();
    if (!options->sexlinked || sex == FEMALE) n += father->visit();
  }
  return n;
}

Uint Person::visitchildren() {
  Uint naff = 0;
  for (Plist *c = children; c != 0; c = c->next)
    if (!options->sexlinked || sex == FEMALE || c->p->sex == FEMALE)
      naff += c->p->visitdown();
  return naff;
}

Uint Person::visitparents() {
  Uint naff = 0;
  if (!founder()) {
    naff += mother->visitup();
    if (!options->sexlinked || sex == FEMALE) naff += father->visitup();
  }
  return naff;
}

Uint Person::visitup() {
  if (visitedup) return 0;
  Uint naff = 0;
  if (dstat != UNKNOWN && !visitedup && !visiteddown) naff++;
  visitedup = true;
  if (dstat != UNKNOWN || founder()) naff += visitchildren();
  naff += visitparents();
  return naff;
}

Uint Person::visitdown() {
  if (visiteddown) return 0;
  Uint naff = 0;
  if (dstat != UNKNOWN && !visitedup && !visiteddown) naff++;
  visiteddown = true;
  naff += visitchildren();
  if (dstat != UNKNOWN) naff += visitparents();
  return naff;
}

void Person::sort(bool down, Person **last) {
  if (!father->processed) father->sort(false,last);
  if (!mother->processed) mother->sort(false,last);
  if (!processed) {
    // put this at end of list
    (*last)->next = this;
    next = 0;
    *last = this;
    processed = true;
  }
  if (down) 
    for (Plist *c = children; c != 0; c = c->next)
      c->p->sort(true, last);
}

void Person::addchild(Person *c) {
  children = new Plist(c,children);
}

void Person::removechild(Person *p) {
  assertcond(children != 0,
             "Trying to remove a child from a person with no children");
  if (p == children->p) {
    Plist *tmp = children->next;
    children->next = 0;
    delete children;
    children = tmp;
  }
  else {
    // Find child before p
    Plist *s;
    for (s = children; s != 0 && s->next->p != p; s = s->next);
    assertinternal(s != 0);
    Plist *tmp = s->next->next;
    s->next->next = 0;
    delete s->next;
    s->next = tmp;
  }
}

void Person::addparents(Person *fath, Person *moth) {
  if (fath == 0 && moth == 0) return;
  assertcond(fath != 0 && moth != 0, "null parents of " + id + "!");
  father = fath;
  mother = moth;
  assertcond(father->sex == MALE, id + " has a female father");
  assertcond(mother->sex == FEMALE, id + " has a male mother");
  father->addchild(this);
  mother->addchild(this);
}

void Person::getspouses(Person *&s1, Person *&s2) const {
  s1 = s2 = 0;
  for (Plist *c = children; c != 0 && s2 == 0; c = c->next) {
    if (sex == MALE) {
      if (s1 == 0) s1 = c->p->mother;
      else if (s1 != c->p->mother) s2 = c->p->mother;
    }
    else {
      if (s1 == 0) s1 = c->p->father;
      else if (s1 != c->p->father) s2 = c->p->father;
    }
  }
}

///////////////////////////////////////////////////////////////////////////
// Family
void Family::countbits() {
  num = numnf = numfc = numf = nqtl = 0;
  for (Person *p = first; p != 0; p = p->next) {
    num++;
    p->removepatbit = options->sexlinked;
    p->removematbit = false;
    p->fc = 0;
    if (p->founder() && (!options->sexlinked || p->sex == FEMALE)) numf++;
    else if (!p->founder()) numnf++;
    if (p->hastraitvalue()) nqtl++;
  }
  for (Person *p = first; p != firstdescendant; p = p->next)
    if (p->children != 0)
      if (p->sex == MALE) p->children->p->removepatbit = true;
      else p->children->p->removematbit = true;
  // Find founder couples
  if (options->foundercouples && !options->sexlinked &&
      !options->sexspecific) findfoundercouples();
  for (Foundercouple *fc = firstfoundercouple; fc != 0; fc = fc->next) {
    assertinternal(fc->wife->children != 0);
    Person *child = fc->wife->children->p;
    if (child->sex == MALE) child->children->p->removepatbit = true;
    else child->children->p->removematbit = true;
  }
  if (numnf == 0) {
    assertcond(num <= 1, "Familiy " + id +
               " consists of more than one unconnected people");
    numbits = 0;
  }
  else numbits = options->sexlinked ? numnf - numf : 2*numnf - numf - numfc;
  assertcond(numbits <= MAXBITS, "Family " + id +
             " has too many bits (" + numbits + ")");
  assertinternal(num > 0);
  if (numbits == 0)
    warning(string("Family ") + id +
            " has 0 bits and is therefor uninformative for linkage");
  numiv = POW2[numbits];
  assertinternal(options->sexlinked || numf + numnf == num);
}

Uint Family::hodgeninfm() {
  Uint ninfm = 0;
  for (Person *p = first; p != 0; p = p->next)
    if (p->dstat == AFFECTED && p->genotyped) {
      for (Person *q = first; q != 0; q = q->next)
        q->visitedup = q->visiteddown = false;
      if (p->ishodgeinformativeup(0)) ninfm++;
    }
  return ninfm;
}

void Family::calckinship() {
  if (kinship == 0) kinship = new Kinship(this);
}

bool Family::hasinformativerelatives(Person *p) {
  return countinformativerelatives(p) > 0;
}

void Family::init() {
  num = numnf = numfc = numf = 0;
  mask = 0;
  first = 0;
  last = 0;
  next = 0;
  lastfounder = 0;
  firstdescendant = 0;
  firstfoundercouple = 0;
  kinship = 0;
}

Family::Family(const string& fid, Uint popidx) :
    id(fid), populationindex(popidx) {
  init();
  // Find qtl trait values
}

Family::Family(const Family &fam) :
    id(fam.id), populationindex(fam.populationindex) {
  init();
  for (Person *p = fam.first; p != 0; p = p->next) addperson(p);
}

Family::~Family() {
  deletelist(first);
  deletelist(firstfoundercouple);
}

void Family::removeuninformativeperson(Person *p) {
  assertinternal(p != 0);
  warning(p->id + " in family " + id +
          " is uninformative and has been discarded");
  removeperson(p);
}

void Family::removeperson(Person *p) {
  assertinternal(p != 0);
  // Update the list of people in the family
  if (p == first) first = p->next;
  else {  
    // Find the person before p in the family
    Person *q;
    for (q = first; q != 0 && q->next != p; q = q->next);
    q->next = p->next;
    if (p == lastfounder) lastfounder = q;
    if (p == firstdescendant) firstdescendant = p->next;
  }
  // Remove p from his parents list of children
  if (p->father != 0) p->father->removechild(p);
  if (p->mother != 0) p->mother->removechild(p);
  // Remove p as a parent from his children
  for (Plist *c = p->children; c != 0; c = c->next) {
    if (p->sex == MALE) c->p->father = 0;
    else c->p->mother = 0;
  }
  delete p;
}

Person *Family::addperson(const string& pid, Sex sex, Diseasestatus ds,
                          Uint liab, Double trval) {
  Person *p = new Person(pid, sex, ds, liab, trval);
  if (last != 0) last->next = p;
  last = p;
  if (first == 0) first = p;
  return p;
}

Person *Family::addperson(Person *p, bool addpar) {
  Person *q = new Person(*p);
  if (last != 0) last->next = q;
  last = q;
  if (first == 0) first = q;
  if (!p->founder()) {
    if (firstdescendant == 0) firstdescendant = q;
    if (addpar)
      q->addparents(findperson(p->father->id), findperson(p->mother->id));
  }
  return q;
}

void Family::organize(const vector<string> &faid, const vector<string> &moid) {
  Person *p = first;
  for (int j = 0; p != 0; j++, p = p->next) {
    bool missf = faid[j] == "0";
    bool missm = moid[j] == "0";
    assertcond(missf == missm, "Only one parent given for person " + 
               p->id + " in family " + id);
    if (!missf && !missm) {
      Person *f = findperson(faid[j]);
      Person *m = findperson(moid[j]);
      assertcond(f != 0 && m != 0, string("Parent of ") + p->id +
                 " not found)");//, family " + id + " skipped\n");
      p->addparents(f,m);
    }
  }
  organize();
}

void Family::organize() {
  putfoundersfirst();
  sort();
  // Remove uninformative phenotypes
  removeuninformativephenotypes();
  // Remove completely uninformative people
  if (!options->nosplitting)
    trim();
}

void Family::removeuninformativephenotypes() {
  for (Person *p = first; p != 0; p = p->next)
    if (p->dstat != UNKNOWN && !hasinformativerelatives(p))
      p->dstat = UNKNOWN;
}

void Family::trim() {
  Person *cur = first;
  while (cur != 0) {
    if (!cur->genotyped && cur->dstat == UNKNOWN && cur->children == 0 &&
        !cur->hastraitvalue()) {
      // cur is an unaffected ungenotyped leaf
      removeuninformativeperson(cur);
      cur = first;
    }
    else if (!cur->founder() &&
             cur->mother->dstat == UNKNOWN && cur->father->dstat == UNKNOWN &&
             cur->mother->children->next == 0 && cur->mother->founder() &&
             cur->father->children->next == 0 && cur->father->founder() &&
             !cur->mother->genotyped && !cur->father->genotyped) {
      // Parents of cur form a "pigtail"
      removeuninformativeperson(cur->mother);
      removeuninformativeperson(cur->father);
      if (firstdescendant == cur) {
        firstdescendant = cur->next;
        lastfounder = cur;
      }
      else {
        // Find the person before cur in the family
        Person *q;
        for (q = first; q != 0 && q->next != cur; q = q->next);
        q->next = cur->next;
        cur->next = first;
        first = cur;
      }
      cur = first;
    }
    else cur = cur->next;
  }
}

Person *Family::findperson(const string& id) {
  Person *p = first;
  while (p != 0 && p->id != id) p = p->next;
  return p;
}

// Put founders first
void Family::putfoundersfirst() {
  if (first == 0) return;
  Person *p = first;
  lastfounder = 0;
  if (first->founder()) lastfounder = first;
  while (p->next) {
    Person *q = p->next; 
    if (q->founder()) {
      if (lastfounder == 0) lastfounder = q;
      p->next = q->next;
      q->next = first;
      first = q;
    }
    else p = p->next;
  }
}

void Family::sort() { // so that children come after parents,
  for (Person *p = first; p != 0; p = p->next)
    p->processed = false;
  lastfounder->next = 0; // Just a list of founders
  Person *last = lastfounder;    //
  for (Person *p = first; p != 0; p = p->next)
    p->processed = true;
  // Don't change lastfounder->next to firstdescendant on following line
  for (Person *p = first; p != lastfounder->next; p = p->next)
    for (Plist *c = p->children; c != 0; c = c->next)
      c->p->sort(true, &last);
  firstdescendant = lastfounder->next;
}

bool Family::connected() {
  if (first == 0) return true;
//  assertcond(first != 0, "Empty family " + id + "!");
  if (first->next == 0) return true;
  Uint n = 0;
  for (Person *p = first; p != 0; p = p->next) {
    p->visitedup = p->visiteddown = false;
    if (options->sexlinked && p->sex == MALE && p->founder() &&
        p->children != 0) {
      bool allchildrenmale = true;
      for (Plist *c = p->children; c != 0 && allchildrenmale; c = c->next)
        if (c->p->sex == FEMALE) allchildrenmale = false;
      if (!allchildrenmale) n++;
    }
    else n++;
  }
  // Find first female
  Person *p = first;
  while (p != 0 && p->sex == MALE) p = p->next;
  if (p == 0) {
    assertinternal(first != 0);
    first->visitedup = true;
    return false;
  }
  else return p->visit() == n;
}  

Uint Family::countinformativerelatives(Person *q) {
  for (Person *p = first; p != 0; p = p->next)
    p->visiteddown = p->visitedup = false;
  return q->visitdown() - (q->dstat == UNKNOWN ? 0 : 1);
}

bool Family::phenotypesconnected() {
  if (first == 0) return true;
  Uint naff = 0;
  for (Person *p = first; p != 0; p = p->next)
    if (p->dstat != UNKNOWN) naff++;
  Person *firstaffected = first;
  while (firstaffected != 0 && firstaffected->dstat == UNKNOWN)
    firstaffected = firstaffected->next;
  if (firstaffected == 0) return true;
  else return naff - 1 == countinformativerelatives(firstaffected);
}

void shufflechildren(Plist *&children, Plist *child) {
  if (children != child) {
    children->p->removepatbit = children->p->removematbit = false;
    Plist *last = children;
    for (; last->next != child && last != 0; last = last->next);
    assertinternal(last != 0);
    last->next = child->next;
    child->next = children;
    children = child;
    children->p->removepatbit = children->p->removematbit = true;
  }
}

void Family::findfoundercouples() {
  // Should delete old founder couples !!!
  firstfoundercouple = 0;
  for (Person *p = first; p != 0; p = p->next) {
    if (p->sex == MALE && p->founder() && !p->genotyped) {
      Person *w1, *w2;
      p->getspouses(w1, w2);
      if (w1 != 0 && w2 == 0 && w1->founder() && !w1->genotyped) {
        Person *h1, *h2;
        w1->getspouses(h1, h2);
        if (h2 == 0) {
          // We have a potential founder couple
          // now see if they have any grandchildren
          bool hasgrandchildren = false;
          for (Plist *c = p->children; c != 0 && !hasgrandchildren; c = c->next)
            hasgrandchildren = c->p->children != 0;
          if (hasgrandchildren) {
            // We have an ungenotyped founder couple that has grandchildren
            numfc++;
            firstfoundercouple = new Foundercouple(p,  w1, firstfoundercouple);
            p->fc = w1->fc = firstfoundercouple;
            // Shuffle grandchildren so that the first child of husband and wife
            // has children
            Plist *wc, *hc;
            for (wc = firstfoundercouple->wife->children,
                   hc = firstfoundercouple->husband->children;
                 wc != 0 && wc->p->children == 0; wc = wc->next, hc = hc->next);
            assertcond(wc->p->children != 0, "Unable to find grandchild of fc");
            shufflechildren(firstfoundercouple->wife->children, wc);
            shufflechildren(firstfoundercouple->husband->children, hc);
            for (Plist *c = w1->children; c != 0; c = c->next)
              c->p->fc = firstfoundercouple;
          }
        }
      }
    }
  }
}

Uint Family::nasymmetricfoundercouples() const {
  Uint res = 0;
  for (Foundercouple *fc = firstfoundercouple; fc != 0; fc = fc->next)
    if (fc->asymmetric()) res++;
  return res;
}

int splitindex = 0;

void Family::split() {
  if (options->nosplitting) return;
  for (Person *p = first; p != 0; p = p->next)
    p->visitedup |= p->visiteddown;
  Family *newfam = new Family(id + "_split" + splitindex);
  splitindex++;
  newfam->next = next;
  next = newfam;
  for (Person *p = first; p != 0; p = p->next) {
    if (options->sexlinked && !p->founder() && !p->visitedup) {
      if (!p->mother->visitedup && p->father->visitedup)
        newfam->findorinsert(p->father);
      else if (!p->father->visitedup && p->mother->visitedup)
        newfam->findorinsert(p->mother);
    }
    if (!p->visitedup) newfam->addperson(p);
  }
  newfam->organize();
  newfam->setup();

  for (Person *p = firstdescendant; p != 0; p = p->next)
    if (p->visitedup && p->mother->visitedup && !p->father->visitedup) {
      if (options->sexlinked) {
        p->father->removechild(p);
        Person *fa = new Person(p->id + "_father", MALE, UNKNOWN, 0,
                                NOTRAITVALUE);
        fa->next = first;
        first = fa;
        fa->visitedup = true;
        fa->addchild(p);
        p->father = fa;
      }
      else
        p->father->visitedup = true;
    }
    else if (p->visitedup && p->father->visitedup && !p->mother->visitedup) {
      p->mother->visitedup = true;
//        if (options->sexlinked) {
//          p->mother->removechild(p);
//          Person *mo = new Person(p->id + "_mother", FEMALE, UNKNOWN, 0,
//                                  NOTRAITVALUE);
//          mo->next = first;
//          first = mo;
//          mo->visitedup = true;
//          mo->addchild(p);
//          p->mother = mo;
//        }
    }
  
  for (Person *p = first; p != 0;) {
    Person *pnext = p->next;
    if (!p->visitedup) removeperson(p);
    p = pnext;
  }
  organize();
  setup();
}

void Family::splitunconnectedphenotypes() {
  if (!phenotypesconnected()) split();
}

Uint Family::countinformative() const {
  Uint ninf = 0;
  for (Person *p = first; p != 0; p = p->next)
    if (p->dstat != UNKNOWN) ninf++;
  return ninf;
}

void Family::findorinsert(Person *p) {
  for (Person *q = first; q != 0; q = q->next)
    if (p->id == q->id) return;
  addperson(p, false);
}

void Family::setup() {
  if (!connected()) {
    warning("Family " + id +
            " is not connected; splitting family for analysis");
    // Loop through unconnected people and add them to new family, and remove
    // them from this one
    split();
  } else {
    removeuninformativephenotypes();
    splitunconnectedphenotypes();
    
    // Count informative people
    Uint ninf = countinformative();
    if (ninf < 2) 
      warning(string("Family ") + id +
              " has uninformative phenotypes and is uninformative for linkage)");
  }
}

void Family::pseudonull(DoubleVec p0) {
  IV v0 = 0;
  IV v0mask = 0;
  for (Person *p = first; p != 0; p = p->next)
    if (p->sex == MALE && p->children != 0)
      for (Plist *c = p->children; c != 0; c = c->next) {
        v0mask |= c->p->patmask;
        v0 |= p->children->p->sex == c->p->sex ? 0 : c->p->patmask;
      }
  zero(p0, numiv);
  p0[v0] = Double(POW2[wt[v0mask]])/Double(numiv);
  expandiv(p0, ~v0mask, numiv);
}

///////////////////////////////////////////////////////////////////////////
// Foundercouple
Foundercouple::Foundercouple(Person *h, Person *w, Foundercouple *n) :
      husband(h), wife(w), mask(0), childrenmask(0), next(n),
      lastgcbit(0), bitbeforefirstgcbit(0) {}

bool Foundercouple::asymmetric() const {
  return husband->dstat != wife->dstat || husband->hastraitvalue() ||
    wife->hastraitvalue();
}

void Foundercouple::swapdstat() {
  Diseasestatus tmp = husband->dstat;
  husband->dstat = wife->dstat;
  wife->dstat = tmp;

  Uint temp = husband->liability;
  husband->liability = wife->liability;
  wife->liability = temp;

  Double tr = husband->traitvalue;
  husband->traitvalue = wife->traitvalue;
  wife->traitvalue = tr;
}
