#include "files.h"
#include "options.h"
#include "map.h"
#include "control.h"
#include "genotypes.h"
#include "vecutil.h"
#include "family.h"

int Marker::nextidx = 0;

const Uint MAXLINE = 10000;
char gtbuf[MAXLINE];

void readgenotypefile(String2Marker &string2marker, String2PN &string2PN) {
  options->genotypefile.open();
  int linenum = 0;
  while (!options->genotypefile.eof()) {
    linenum++;
    options->genotypefile >> gtbuf;
    PN *pn = string2PN.get(gtbuf, true);
    options->genotypefile >> gtbuf;
    Marker *m = string2marker.get(gtbuf, true);
    int r1, r2;
    options->genotypefile >> r1 >> r2 >> ws;
    if (options->genotypefile.fail())
      fatal(string("Illegal value at line ") + linenum + " in " +
            options->genotypefile.name);
    pn->add(m->idx, r1, r2);
  }
  options->genotypefile.close();
}

Chromosome unknownchromosome("unknown");

void readmapfile(String2Marker &string2marker,
                 String2Chromosome &string2chromosome) {
  options->mapfile.open();
  options->mapfile >> ws;
  char lastchromosome[MAXLINE] = "";
  Chromosome *ch = 0;
  while (!options->mapfile.eof()) {
    options->mapfile >> gtbuf;
    if (gtbuf[0] == '%') continue;
    if (strcmp(gtbuf, lastchromosome) != 0) {
      ch = string2chromosome.get(gtbuf, false);
      assertcond(ch != 0, string("Chromosome ") + gtbuf +
                 " in map file is not in chromosomefile");
      strcpy(lastchromosome, gtbuf);
    }
    options->mapfile >> gtbuf;
    Marker *m = string2marker.get(gtbuf);
    assertcond(m == 0 || m->next == 0,
               string("Marker ") + gtbuf + " repeated in map!");
    if (m != 0) {
      // Check if marker's location is to be imputed
      for (Uint i = 0; i < options->modelmarkers.size() &&
             ch != &unknownchromosome; i++)
        if (options->modelmarkers[i] == gtbuf) {
          ch = &unknownchromosome;
          strcpy(lastchromosome, "unknown");
        }
      options->mapfile >> m->location;
      if (options->sexspecific)
        options->mapfile >> m->location_female;
      ch->add(m);
    }
    options->mapfile.ignore(MAXLINE, '\n');
    options->mapfile >> ws;
  }
  options->mapfile.close();
}

void readfreqfile(String2Marker &string2marker) {
  options->freqfile.open();
  options->freqfile >> ws;
  char lastmarker[MAXLINE] = "";
  Marker *m = 0;
  while (!options->freqfile.eof()) {
    options->freqfile >> gtbuf;
    if (strcmp(gtbuf, lastmarker) != 0) {
      m = string2marker.get(gtbuf);
      strcpy(lastmarker, gtbuf);
    }
    if (m != 0) {
      int rep;
      options->freqfile >> rep;
      Float freq;
      options->freqfile >> freq;
      m->rep2freq[rep] = freq + options->controlprior;
      m->controlsavailable = true;
    }
    options->freqfile.ignore(MAXLINE, '\n');
  }
  options->freqfile.close();
}

void readaffectionfile(String2PN &string2pn) {
  options->affectionfile.open();
  int linenum = 0;
  while (!options->affectionfile.eof()) {
    linenum++;
    options->affectionfile >> gtbuf;
    PN *pn = string2pn.get(gtbuf, true);
    int tmpint;
    options->affectionfile >> tmpint;
    if (options->affectionfile.fail())
      fatal(string("Illegal disease status in affection file at line ") +
            linenum);
    if (tmpint >= 2) tmpint = 2;
    else if (tmpint <= 0) tmpint = 0;
    else tmpint = 1;
    pn->dstat = Diseasestatus(tmpint);
    options->affectionfile >> ws;
  }
  options->affectionfile.close();
}

void readchromosomefile(String2Chromosome &string2chromosome) {
  options->chromosomefile.open();
  while (!options->chromosomefile.eof()) {
    options->chromosomefile >> gtbuf;
    Chromosome *ch = string2chromosome.get(gtbuf, true);
    options->chromosomefile >> ch->length;
    assertcond(ch->length > 0, string("Illegal chromosome length ") +
               ch->length + " for " + ch->id);
    if (options->sexspecific) {
      options->chromosomefile >> ch->length_female;
      assertcond(ch->length_female > 0,
                 string("Illegal female chromosome length ") +
                 ch->length_female + " for " + ch->id);
    }
    options->chromosomefile >> gtbuf;
    switch (gtbuf[0]) {
      case 'a':
      case 'A':
        ch->type = AUTOSOMAL; break;
      case 's':
      case 'S':
        ch->type = SEXLINKED; break;
      default: fatal(string("Unknown chromosome type ") + gtbuf);
    }
    options->chromosomefile.ignore(MAXLINE, '\n');
    options->chromosomefile >> ws;
  }
  options->chromosomefile.close();    
}

void readoldfreqfile(String2Marker &string2marker) {
  options->oldfreqfile.open();
  char lastmarker[MAXLINE] = "";
  Marker *m = 0;
  while (!options->oldfreqfile.eof()) {
    options->oldfreqfile >> gtbuf;
    if (strcmp(gtbuf, lastmarker) != 0) {
      m = string2marker.get(gtbuf);
      strcpy(lastmarker, gtbuf);
    }
    if (m != 0) {
      int rep;
      options->oldfreqfile >> rep >> gtbuf;
      Uint allele;
      options->oldfreqfile >> allele;
      assertinternal(m->rep2allele.find(rep) == m->rep2allele.end());
      m->rep2allele[rep] = allele;
    }
    options->oldfreqfile.ignore(MAXLINE, '\n');
  }
  options->oldfreqfile.close();
}

void Control::inputdecode() {
  readgenotypefile(string2marker, string2pn);
  readchromosomefile(string2chromosome);
  readmapfile(string2marker, string2chromosome);
  readfreqfile(string2marker);
  if (options->affectionfile.assigned()) readaffectionfile(string2pn);
  if (options->oldfreqfile.assigned()) readoldfreqfile(string2marker);
  Infile prefile;
  prefile.setname(options->prefile[0]);
  getpre(prefile, 0, 0, 0);
}

void Control::initializechromosome(const string &id) {
  message("");
  message("Analysing chromosome " + id);
  message("");
  // Set subdir for outputfiles to the id of the chromosome
  if (options->chromolist->next != 0 || options->chromolist->all)
    File::setsubdir(id);
  Chromosome *ch = string2chromosome.get(id.c_str());
  assertcond(ch != 0, "Unable to find chromosome " + id + " in map file");
  map.reset();
  map.setunit(CENTIMORGAN);
  map.init(ch->nmarkers + (options->addendmarkers ? 2 : 0));
  assertcond(ch->nmarkers > 0, "Chromosome " + id + " has no markers");
  // Compute marker distances
  Uint numendmarkers = (options->addendmarkers ? 1 : 0);
  Uint midx = numendmarkers;
  const Float MINDIST = 0.05;
  FloatVec dist = 0;
  FloatVec dist_female = 0;
  Uint numdist = ch->nmarkers + (options->addendmarkers ? 1 : -1);
  if (numdist > 0) {
    NEWVEC(Float, dist, numdist);
    if (options->sexspecific)
      NEWVEC(Float, dist_female, numdist);
  }
  if (options->addendmarkers) {
    dist[0] = ch->firstmarker->location;
    if (options->sexspecific)
      dist_female[0] = ch->firstmarker->location_female;
  }
  for (Marker *m = ch->firstmarker; m != 0 && m->next != 0; m = m->next,
         midx++) {
    dist[midx] = m->next->location - m->location;
    if (options->sexspecific)
      dist_female[midx] = m->next->location_female - m->location_female;
  }
//  assertcond(options->markerdistzero || dist[midx] > 0, "Markers " + m->id +
//              " and " + m->next->id + " are in the same position " +
//              "(use \"MARKERDISTZERO on\" to allow)");
  if (options->addendmarkers) {
    dist[ch->nmarkers] = max(ch->length - ch->lastmarker->location, 0.0);
    if (options->sexspecific)
      dist_female[ch->nmarkers] =
        max(ch->length_female - ch->lastmarker->location_female, 0.0);
  }
  if (!options->markerdistzero) {
    for (midx = 0; midx < ch->nmarkers + (options->addendmarkers ? 1 : -1);
         midx++) {
      if (dist[midx] == 0) dist[midx] = MINDIST;
      if (options->sexspecific)
        if (dist_female[midx] == 0) dist_female[midx] = MINDIST;
    }
  }
  if (options->sexspecific) {
    map.adddist(dist, 1);
    map.adddist(dist_female, 2);
  }
  else
    map.adddist(dist, 0);
  DELETEVEC(dist);
  if (options->sexspecific)
    DELETEVEC(dist_female);
  // Assign allele numbers to repeats and alleles to people
  FloatVec unif2;
  NEWVEC(Float, unif2, 2);
  unif2[0] = unif2[1] = .5;
  if (options->addendmarkers) {
    map.addmarker(0, 2, unif2, "START" + ch->id);
    IntVec orig;
    NEWVEC(int, orig, 2);
    orig[0] = 1;
    orig[1] = 2;
    map.origalleles.push_back(orig);
  }
  
  midx = numendmarkers;
  Outfile controlfrq("control.frq");
  controlfrq.open();
  for (Marker *m = ch->firstmarker; m != 0; midx++) {
    // Create frequencies
    for (String2PN::container::iterator pn = string2pn.rep.begin();
         pn != string2pn.rep.end(); pn++) {
      if (pn->second->genotyped()) {
        IntVec als = pn->second->get_rep(m->idx);
        if (als != 0 && als[0] != options->unknownrepeat) {
          if (m->rep2freq[als[0]] == 0.0) m->addallele(als[0]);
          if (m->rep2freq[als[1]] == 0.0) m->addallele(als[1]);
        }
      }
    }
    if (!m->controlsavailable) {
      warning("No control frequencies available for " + m->id +
              " (using data as controls)");
      for (String2PN::container::iterator pn = string2pn.rep.begin();
           pn != string2pn.rep.end(); pn++) {
        IntVec als = pn->second->get_rep(m->idx);
        if (pn->second->genotyped() && als != 0 &&
            als[0] != options->unknownrepeat) {
          assertinternal(als[1] != options->unknownrepeat);
          m->rep2freq[als[0]] += 1.0;
          m->rep2freq[als[1]] += 1.0;
        }
      }
    }
    // Map repeats into alleles
    m->assignrep2allele();
    assertcond(m->rep2freq.size() == m->rep2allele.size(),
               m->id + " incorrectly processed");    
    // Add mapping from allegro alleles to original alleles
    IntVec orig;
    NEWVEC(int, orig, m->rep2allele.size());
    for (Int2int::iterator al = m->rep2allele.begin();
         al != m->rep2allele.end(); al++) {
      assertinternal(al->second <= m->rep2allele.size());
      orig[al->second - 1] = al->first;
    }
    map.origalleles.push_back(orig);

    if (midx < map.num) {
      // Add marker to map
      FloatVec freq = m->getfreq(&controlfrq);
      map.addmarker(midx, m->rep2allele.size(), freq, m->id);
      DELETEVEC(freq);
    }
    if (m->next == 0 && midx < map.num) {
      midx++;
      m = unknownchromosome.firstmarker;
    }
    else m = m->next;
  }

  controlfrq.close();
  if (options->addendmarkers) {
    map.addmarker(map.num - 1, 2, unif2, "END" + ch->id);
    IntVec orig;
    NEWVEC(int, orig, 2);
    orig[0] = 1;
    orig[1] = 2;
    map.origalleles.push_back(orig);
  }

  // Write map
  Outfile mapfile("map");
  mapfile.open();
  assertinternal(mapfile.is_open());
  for (Uint m = numendmarkers; m < map.num - numendmarkers; m++)
    mapfile << ch->id << "\t" << map.markername[m] << "\t" << map.markerpos[0][m]
            << "\n";
  mapfile.close();

  options->sexlinked = ch->type == SEXLINKED;
  IntVec nogt;
  NEWVEC(int, nogt, 2);
  nogt[0] = nogt[1] = options->unknownrepeat;
  assertinternal(options->modelmarkers.size() == unknownchromosome.nmarkers);
  for (Family *fam = firstorig; fam != 0; fam = fam->next)
    for (Person *p = fam->first; p != 0; p = p->next) {
      // Set liability classes
      p->liability = options->sexlinked ? 1 - p->sex : 0;
      // Set genotypes for genotyped people
      PN *q = string2pn.get(p->id.c_str());
      if (q != 0 && q->genotyped()) {
        p->allocgenotypes(map.num + options->modelmarkers.size());
        midx = numendmarkers;
        Marker *m = ch->firstmarker;
        while (midx < map.num + options->modelmarkers.size()) {
          assertinternal(m != 0);
          IntVec als = q->get_rep(m->idx);
          if (als == 0) als = nogt;
          if (als[0] != options->unknownrepeat) {
            Int2int::iterator fr = m->rep2allele.find(als[0]);
            assertinternal(fr != m->rep2allele.end());
            fr = m->rep2allele.find(als[1]);
            assertinternal(fr != m->rep2allele.end());
            assertinternal(m->rep2allele[als[0]] != 0 &&
                           m->rep2allele[als[1]] != 0);
            p->gen[0][midx] = m->rep2allele[als[0]];
            p->gen[1][midx] = m->rep2allele[als[1]];
            p->genotyp[midx] = true;
          }
          else {
            p->gen[0][midx] = p->gen[1][midx] = ALLELEUNKNOWN;
            p->genotyp[midx] = false;
          }
          // Increment midx and m
          m = m->next;
          if (midx < map.num && m == 0) {
            midx += 1 + numendmarkers;
            m = unknownchromosome.firstmarker;
          }
          else midx++;
        }
      }
    }
}

void Chromosome::add(Marker *m) {
  for (Marker *n = firstmarker; n != 0; n = n->next)
    assertcond(n->id != m->id, "Marker " + m->id + " is duplicated in map!");
  if (firstmarker == 0) lastmarker = firstmarker = m;
  else if (m->location >= lastmarker->location) {
    lastmarker->next = m;
    lastmarker = m;
  }
  else if (m->location <= firstmarker->location) {
    m->next = firstmarker;
    firstmarker = m;
  }
  else {
    Marker *n;
    // Find next marker before this one
    for (n = firstmarker; n->next != 0 && n->next->location <= m->location;
         n = n->next);
    m->next = n->next;
    n->next = m;
    if (m->next == 0) lastmarker = m;
  }
  nmarkers++;
}

PN::~PN() {
  DELETEMAT(rep);
}

void PN::add(int idx, int r1, int r2) {
  while (idx >= nrep) {
    IntMat newrep;
    NEWMAT(int, newrep, nrep + repinc, 2);
    copyval(newrep[nrep], options->unknownrepeat, 2*repinc);
    if (nrep > 0) {
      copyvec(newrep[0], rep[0], 2*nrep);
      DELETEMAT(rep);
    }
    rep = newrep;
    nrep += repinc;
  }
  rep[idx][0] = r1;
  rep[idx][1] = r2;
}

FloatVec Marker::getfreq(ostream *out) {
  FloatVec freq;
  NEWVEC(Float, freq, rep2allele.size());
  for (Int2int::iterator al = rep2allele.begin();
       al != rep2allele.end(); al++) {
    assertinternal(al->second > 0 && al->second - 1 < rep2allele.size());
    freq[al->second - 1] = rep2freq[al->first];
    // Write control frequencies to file
    if (out != 0)
      (*out) << id << "\t" << al->first << "\t"
             << rep2freq[al->first] << "\t" << al->second << "\n";
  }
  assertcond(rep2freq.size() == rep2allele.size(),
             id + " incorrectly processed!");
  return freq;
}

void Control::getmarkerallelefreq(const string &markerid, FloatVec &freq) {
  Marker *m = string2marker.get(markerid.c_str(), false);
  assertinternal(m != 0);
  freq = m->getfreq();
}

Uint Control::getmarkerindex(const string &markerid) {
  Uint midx = 0;
  Marker *m;
  for (m = unknownchromosome.firstmarker; m != 0; m = m->next, midx++)
    if (m->id == markerid) break;
  assertinternal(m != 0);
  return map.num + midx;
}

void Marker::assignrep2allele() {
  // Find smallest unassigned repeat
  const int NOREPEAT = 9999999;
  int repeat = NOREPEAT;
  for (Int2float::iterator al = rep2freq.begin(); al != rep2freq.end(); al++)
    if (rep2allele.find(al->first) == rep2allele.end() &&
        al->first < repeat) repeat = al->first;
  a = rep2allele.size() + 1;
  if (repeat != NOREPEAT) {
    rep2allele[repeat] = a++;
    while (rep2freq.size() > rep2allele.size()) {
      repeat++;
      if (rep2freq.find(repeat) != rep2freq.end() &&
          rep2allele.find(repeat) == rep2allele.end())
        rep2allele[repeat] = a++;
      assertcond(repeat < 10000, string("Repeat ") + repeat + " too long!");
    }
  }
}
