#include <iostream>
#include <iomanip>
#include "basic.h"
#include "map.h"
#include "files.h"
#include "options.h"
#include "vecutil.h"
#include "haldane.h"

Map::Map() {
  start();
}

void Map::start() {
  numposition = num = 0;
  numallele.clear();
  pi.clear();
  for (Uint i = 0; i <= 2; i++) {
    markerpos[i].clear();
    theta[i].clear();
    position[i].clear();
  }
  leftmarker.clear();
  markername.clear();
  shouldfindp.clear();
  inbetween.clear();
  while (origalleles.size() > 0) {
    DELETEVEC(origalleles.back());
    origalleles.pop_back();
  }
  unit = options->unit;
}

Map::~Map () {
  deallocate();
}

void Map::readstepfile(Infile& f) {
  const Float eps = 1e-4;
  vector<Float> stp;

  f.open();
  while (!f.eof()) {
    Float st;
    f >> st >> ws;
    stp.push_back(st);
    assertcond(f.eof() || f.good(), "Illegal number in stepfile " + f.name);
    if (stp.size() > 1)
      assertcond(stp[stp.size() - 1] > stp[stp.size() - 2],
                 "Numbers in stepfile " + f.name + "not increasing");
  }
  assertcond(stp.size() > 0, "Empty stepfile");
  allocate(stp.size() + num);
  Uint is = 0;
  Uint im = 0;
  Uint j = 0;
  int leftmrk = -1;
  for (j = 0; is < stp.size() || im < num; j++) {
    if (is < stp.size() && im < num && fabs(markerpos[0][im] - stp[is]) < eps) {
      leftmrk = im;
      for (Uint n = 0; n < nsex; n++) position[n][j] = markerpos[n][im];
      inbetween[j] = false;
      im++;
      is++;
    }
    else if (im >= num || (is < stp.size() && stp[is] < markerpos[0][im])) {
      assertinternal(is < stp.size());
      for (Uint n = 0; n < nsex; n++) position[n][j] = stp[is];
      inbetween[j] = true;
      is++;
    }
    else {
      assertinternal(im < num);
      leftmrk = im;
      shouldfindp[im] = false;
      inbetween[j] = false;
      for (Uint n = 0; n < nsex; n++) position[n][j] = markerpos[n][im];
      im++;
    }
    leftmarker[j] = leftmrk;
  }
  numposition = j;
  f.close();
}  

void Map::init(Uint N) {
  num = N;
  numallele.resize(num);
  for (Uint i = 0; i <= 2; i++) {
    markerpos[i].resize(num);
    theta[i].resize(num); 
  }
  shouldfindp.resize(num);
  //  pi.resize(num);
  markername.resize(num);
}

void Map::allocate(Uint n) {
  for (Uint i = 0; i <= 2; i++) position[i].resize(n);
  inbetween.resize(n);
  leftmarker.resize(n);
}

void Map::reset() {
  deallocate();
  start();
}

void Map::deallocate() {
  if (pi.size() != 0) 
    for (Uint popidx = 0; popidx < pi.size(); popidx++) {
      for (Uint gam = 0; gam < pi[popidx].size(); gam++)
        DELETEVEC(pi[popidx][gam]);
      pi[popidx].clear();
    }
  pi.clear();
  leftmarker.clear();
  inbetween.clear();
  shouldfindp.clear();
  for (Uint i = 0; i <= 2; i++) {
    position[i].clear();
    markerpos[i].clear();
    theta[i].clear();
  }
  markername.clear();
  numallele.clear();
}

void Map::addmarker(Uint m, Uint nump, FloatVec p, const string &mn) {
  assertinternal(m < num);
  numallele[m] = nump;
  markername[m] = mn;
  addpopulationfreq(m, 0, nump, p);
}

void Map::addpopulationfreq(Uint m, Uint popidx, Uint nump, FloatVec p) {
  if (popidx >= pi.size()) {
    pi.resize(popidx + 1);
    pi[popidx].resize(num);
  }
  NEWVEC(Float, pi[popidx][m], nump);
  Float normconst = sum<Float>(p, nump);
  assertcond(normconst > 0 || nump == 0,
             "No possible alleles at marker " + markername[m]);
  for (Uint j = 0; j < nump; j++)
    pi[popidx][m][j] = p[j]/normconst;
}

void Map::maxsteplength() {
  numposition = num;
  IntVec nstep;
  NEWVEC(int, nstep, num - 1);
  for (Uint i = 0; i < num - 1; i++) {
    Float distance = markerpos[0][i + 1] - markerpos[0][i];
    nstep[i] = int(distance/options->maxsteplength);
    numposition += nstep[i];
  }
  allocate(numposition);
  Uint j = 0;
  for (Uint n = 0; n < nsex; n++) {
    for (Uint i = 0; i < num - 1; i++) {
      Float distance = markerpos[n][i + 1] - markerpos[n][i];
      for (int k = 0; k <= nstep[i]; k++) {
        position[n][j + k] = markerpos[n][i] + k*distance/(nstep[i] + 1);
        leftmarker[j + k] = i;
        inbetween[j + k] = k > 0;
      }
      j += nstep[i] + 1;
    }
    position[n][j] = markerpos[n][num - 1];
  }
  assertinternal(j == numposition - 1);
  leftmarker[j] = num - 1;
  inbetween[j] = false;
  DELETEVEC(nstep);
}

//    void Map::readposfromprob() {
//    Read marker positions from the first probfile that is assigned
//    assertinternal(options->firstmodelopt != 0);
//    Uint sc = options->firstmodelopt->score, pt = options->firstmodelopt->pt;
//    Uint nlines = 1000;
//    Double *pos = new Double[nlines];
//    numposition = 0;
//    assertinternal(pt != npt && sc != nscore);
//    options->probfilein[sc][pt].open();
//    string famid, fid;
//    options->probfilein[sc][pt] >> famid >> ws;
//    Double lastpos = -1;
//    while (!options->probfilein[sc][pt].eof() && (famid == fid || lastpos == -1)) {
//      Double p, dum1, dum2;
//      options->probfilein[sc][pt] >> p >> dum1 >> dum2 >> ws;
//      assertcond(options->probfilein[sc][pt].eof() ||
//                 options->probfilein[sc][pt].good(),
//                 "Illegal format of nullprob file " +
//                 options->probfilein[sc][pt].name);
//      if (lastpos != p) {
//        lastpos = pos[numposition] = p;
//        numposition++;
//        if (nlines < numposition) {
//          enlarge(pos, nlines, 1000);
//          nlines += 1000;
//        }
//      }
//      options->probfilein[sc][pt] >> fid >> ws;
//    }
//    options->probfilein[sc][pt].close();
//    allocate(numposition);
//    copyvec(position, pos, numposition);
//    copyval(inbetween, true, numposition);
//    copyval(leftmarker, -1, numposition);
//    delete [] pos;
// }

void Map::fixedstep() {
// find position, leftmarker, shouldfindp, inbetween for fixed no.
// of steps between markers
  numposition = num + (num - 1)*(options->steps - 1);
  allocate(numposition);
  Uint nstep = options->steps - 1;
  Uint j = 0;
  for (Uint n = 0; n < nsex; n++) {
    j = 0;
    for (Uint i = 0; i < num - 1; i++) {
      Float distance = markerpos[n][i + 1] - markerpos[n][i];
      Uint cstep = distance == 0 ? 1 : nstep;
      for (Uint k = 0; k <= cstep; k++) {
        position[n][j + k] = markerpos[n][i] + k*distance/(cstep + 1);
        leftmarker[j + k] = i;
        inbetween[j + k] = k > 0;
      }
      j += cstep + 1;
    }
    position[n][j] = markerpos[n][num - 1];
  }
  leftmarker[j] = num - 1;
  inbetween[j] = false;
}

void Map::adddist(FloatVec dist, int isex) {
// isex = 0 for sex-averaged, 1 for male, 2 for female.  add the
// intermarker distance and finish initialization Should be called
// once with isex=0 or twice, first with isex=1 and the with
// isex=2. The first call is used to determine whether unit is cM or
// recomb (unless the unit is specified in options).
  nsex = options->sexspecific ? 3 : 1;
  if (unit == DEFAULT && isex <= 1) {
    unit = RECOMBINATION;
    for (Uint i = 0; i < num - 1; i++)
      if (dist[i] > 0.5) {
        unit = CENTIMORGAN;
        break;
      }
  }
  if (unit == RECOMBINATION)
    for (Uint i = 0; i < num - 1; i++)
      theta[isex][i] = dist[i];
  else
    for (Uint i = 0; i < num - 1; i++)
      theta[isex][i] = recombfraccent(dist[i]);
  markerpos[isex][0] = 0;
  if (isex == 0) markerpos[0][0] = 0;
  shouldfindp[0] = true;
  for (Uint i = 1; i < num; i++) {
    markerpos[isex][i] = markerpos[isex][i - 1] + centimorgan(theta[isex][i - 1]);
    shouldfindp[i] = true;
  }
  if (isex == 2)
    for (Uint i = 0; i < num; i++) {
      markerpos[0][i] = (markerpos[1][i] + markerpos[2][i])/2;
      theta[0][i] = 0; // not used in this case
    }
  else if (isex == 0) {
    for (Uint i = 0; i < num; i++)
      markerpos[1][i] = markerpos[2][i] = markerpos[0][i];
  }
  if (isex == 2 || isex == 0) 
    createpos();
}

void Map::createpos() {
  if (options->maxsteplength > 0)
    maxsteplength();
  else if (options->stepfile.assigned())
    readstepfile(options->stepfile);
//  else if (options->prefile.size() == 0 && !options->datfile.size() == 0)
//    readposfromprob();
  else
    fixedstep();
//  for (i = 0; i <= ######KRISTJAN LAGAR
}

extern Double runif();

Allele Map::randomallele(Uint gam) const {
  Double rnd = runif();
  Double sum = 0;
  for (Uint j = 0; j < numallele[gam] - 1; j++) {
    sum = sum + pi[0][gam][j]; // Note change to allow simulation from multiple pops
    if (sum > rnd) return j + 1;
  }
  return numallele[gam];
}
