#ifndef _MATRIX
#define _MATRIX
#include <fstream>
#include <stdio.h>
#include "basic.h"

class Probability;

template <class T, class Tp>
class Matrix {
public:
  enum Storagetype {MEMORY, DISK, RECALCQ, RECALCLQ};
protected:
  Storagetype storagetype;
  Tp x;
  Uint nx;
  fstream *f;
  Uint currentrow;
  bool familyfitsinmemory;
  string swapfileprefix;
  void init(Tp buf, Uint n)
    {x = buf; nx = n; f = 0; probability = 0; currentrow = -1; familyfitsinmemory = false;};
  
public:
  // Store matrix in memory
  Matrix(Tp buf, Uint nbuf) : storagetype(MEMORY) {init(buf, nbuf);}
  // Store matrix in file
  Matrix(Tp buf, Uint nbuf, const string &filename, const string &prefix);
  // Recalculate distribution, either from singlelocus or lqhat
  Matrix(Tp buf, Uint nbuf, Storagetype st) :
      storagetype(st) {init(buf, nbuf);};
  ~Matrix();
  
  // Returns the row-th row in the matrix
  Tp getrow(Uint row, bool recalc);
  
  // If matrix is stored in a file a call to this function causes the
  // contents of buf to be written to file. This call only has an effect
  // storagetype is DISK.
  void store(Uint gam);

  // Notifies all the instances of the class that a new family is being
  // processed
  void nextfam(Uint mem, IV N);

  Probability *probability;

private:
//  static Family* fam;
  static IV numiv;
  string fn;

  // If stored in memory mat[gam] returns x + gam*ncols

  // If stored in a binary file f then mat[gam] causes the appropriate part
  // of f to be read into buf and returns buf

  // If to be recalculated from a single locus distribution then mat[gam]
  // returns the same value as probability->findq(gam).

  // If to be recalculated from lqhat then mat[gam] returns the same value as
  // probability->lqfromlqhat(gam).

  void openfile(Uint gam, ios::openmode mode);
  void closefile();
};

typedef Matrix<Float, FloatVec> Floatmatrix;
typedef Matrix<IV, IVVec> IVmatrix;
typedef Matrix<int, IntVec> Intmatrix;

template <class T, class Tp>
Matrix<T, Tp>::Matrix(Tp buf, Uint nbuf, const string &filename,
                      const string &prefix) :
    storagetype(DISK) {
  fn = filename;
  swapfileprefix = prefix;
  init(buf, nbuf);
  f = 0;
}

template <class T, class Tp>
Matrix<T, Tp>::~Matrix() {
  for (int i = 0; i < 300; i++) {
    string curfn = fn + "/" + swapfileprefix + i;
    remove(curfn.c_str());
  }
}

template <class T, class Tp>
void Matrix<T, Tp>::closefile() {
  if (f != 0) {
    f->close();
    delete f;
    f = 0;
  }
}

template <class T, class Tp>
void Matrix<T, Tp>::openfile(Uint gam, ios::openmode mode) {
  string curfn = fn + "/" + swapfileprefix + gam;
  f = new fstream(curfn.c_str(), mode | ios::binary);
  assertcond(f != 0, "Unable to open swapfile " + curfn);
}

template <class T, class Tp>
void Matrix<T, Tp>::nextfam(Uint mem, Uint N) {
  familyfitsinmemory = nx >= mem;
  numiv = N;
}

template <class T, class Tp>
void Matrix<T, Tp>::store(Uint gam) {
  currentrow = gam;
  if (storagetype == DISK && !familyfitsinmemory) {
    openfile(gam, ios::out);
//    f->seekp(ios::beg);
    f->write((char *)(x VEC_GET_DATA), numiv*sizeof(T));
    assertcond(!f->fail(), "Unable to write swapfile to disk");
    closefile();
  }
}

#endif // _MATRIX
