#include <math.h>
#include "maximize.h"
#include "warning.h"
#include "utils.h"
#include "files.h"
#include "options.h"

#define GOLDENSECTION 1.61803398875
#define TOL (1.0e-6)

Maximizor::Maximizor(Double xmin, Double xmax, Uint maxit, Uint gridpoints,
                     bool oob) :
    amin(xmin), amax(xmax), maxiter(maxit), gridsize(gridpoints),
    gooutofbounds(oob) {
  NEWVEC(Double, grid, gridsize);
  NEWVEC(Double, gridval, gridsize);
  Double steplength = (amax - amin)/(gridsize - 1);
  for (Uint i = 0; i < gridsize; i++) grid[i] = amin + i*steplength;
}

Maximizor::~Maximizor() {
  DELETEVEC(grid);
  DELETEVEC(gridval);
}

void Maximizor::goldenstep(Double &a1, Double &a2, Double &a3, Double a4,
                           Double &f2, Double &f3) {
  a1 = a2;
  a2 = a3;
  a3 = a4 - (a2 - a1);
  f2 = f3;
  f3 = f(a3, pos);
}

void Maximizor::goldensearch(Double a1, Double a2, Double a4, Double f2) {
  Double fold1 = f2 - 1, fold2 = fold1 - 1, f3;
  Double a3;
  if (fabs(a2 - a1) > fabs(a4 - a2)) {
    a3 = a2;
    a2 = a1 + a4 - a3;
    f3 = f2;
    f2 = f(a2, pos);
  }
  else {
    a3 = a4 - (a2 - a1);
    f3 = f(a3, pos);
  }
  while (fabs(max_(f2, f3) - fold2) > TOL && niter < maxiter) {
    fold2 = fold1;
    fold1 = min_(f2, f3);
    if (f3 > f2) goldenstep(a1, a2, a3, a4, f2, f3);
    else goldenstep(a4, a3, a2, a1, f3, f2);
    niter++;
  }
  if (f2 > f3) updatemaximum(a2, f2);
  else updatemaximum(a3, f3);
}

void Maximizor::goldenbound(Double a1, Double a4, Double f1) {
  Double a2 = a4, f2 = f1 - 1;
  while (niter < maxiter && TOL < f1 - f2) {
    if (gooutofbounds) {
      a2 = a1;
      f2 = f1;
      a1 = a2 - (a4 - a2)*GOLDENSECTION/(1 + GOLDENSECTION);
      f1 = f(a1, pos);
    }
    else {
      a4 = a2;
      a2 = a1 + (a4 - a1)/(1 + GOLDENSECTION);
      f2 = f(a2, pos);
    }
    niter++;
  }
  if (niter < maxiter && f2 >= f1)
    goldensearch(a1, a2, a4, f2);
  else
    updatemaximum(a1, f1);
}

int sign(Double x) {
  if (x < 0) return -1;
  else if (x > 0) return 1;
  else return 0;
}

void Maximizor::updatemaximum(Double a, Double y) {
  if (y < ymin) return;
  if (maximafound && options->warnonmultiplemaxima) 
    warning(string("Multiple maxima at ") + pos);
  if (!maximafound || y > ystar) {
    astar = a;
    ystar = y;
    maximafound = true;
  }
  if (niter > maxiter)
    warning(string("Maximum number of iterations(") + maxiter +
            ") exceeded at " + pos);
}

void Maximizor::maximize(Double ym, Double xm, likelihoodfunction fun,
                         Uint posit, Double &x, Double &y) {
  niter = 0;
  pos = posit;
  f = fun;
 
  enum Slope {NEGATIVE = -1, ZERO = 0, POSITIVE = 1};
  Slope slope = ZERO;
  
  if (grid[0] == xm) gridval[0] = ym;
  else gridval[0] = f(grid[0], pos);
  if (xm < grid[0] || xm > grid[gridsize - 1]) {
    xm = grid[0];
    ym = gridval[0];
  }
  ystar = ymin = ym;
  astar = xmin = xm;
  maximafound = false;
  bool uninformative = true;
  bool lflat = false;
  for (Uint i = 1; i < gridsize; i++) {
    if (grid[i] == xm) gridval[i] = ym;
    else gridval[i] = f(grid[i], pos);
    Slope curslope = Slope(sign(gridval[i] - gridval[i - 1]));
    if (i == 1) slope = curslope;
    uninformative &= slope == ZERO;
    if (slope == ZERO && curslope == NEGATIVE) 
      // Interior maxima between i - 2 and i - 1
      goldenstart(grid[i - 2], grid[i - 1], gridval[i - 1]);
    else if (slope == POSITIVE  && curslope == ZERO) 
      // Interior maxima between i - 1 and i
      goldenstart(grid[i - 1], grid[i], gridval[i - 1]);
    else if (slope == POSITIVE && curslope == NEGATIVE)
      // Interior maximan betwee i - 2 and i
      goldenstart(grid[i - 2], grid[i - 1], grid[i], gridval[i - 1]);
    else if (slope == ZERO && curslope == ZERO && !uninformative)
      lflat = true;
    slope = curslope;
  }
//  if (lflat) warning(string("Likelihood is very flat at ") + pos);
  if (gridval[0] > gridval[1])
    // Maxima at left boundry
    goldenbound(grid[0], grid[1], gridval[0]);
  if (gridval[gridsize - 1] > gridval[gridsize - 2]) {
    // Maxima at right boundry
    goldenbound(grid[gridsize - 1], grid[gridsize - 2], gridval[gridsize - 1]);
  }

  if (maximafound || uninformative) {
    x = astar;
    y = ystar;
  }
  else {
    x = xmin;
    y = ymin;
//    warning(string("Unable to find a higher liklihood than that of the null at ") + pos);
  }
}

void Maximizor::goldenstart(Double a1, Double b, Double a4, Double fb) {
  Double a2 = a1, a3 = a4;
  Double flast, f2, f3;
  flast = f2 = f3 = fb - 1;
  while (niter < maxiter && max_(f2, f3) <= fb && fb - min_(f2, f3) > TOL) {
    a1 = a2;
    a4 = a3;
    a2 = a1 + (a4 - a1)/(1 + GOLDENSECTION);
    a3 = a4 - (a2 - a1);
    f2 = f(a2, pos);
    f3 = f(a3, pos);
    niter++;
  }
  if (fb < max_(f2, f3)) {
    if (f2 < f3) goldensearch(a2, a3, a4, f3);
    else goldensearch(a1, a2, a3, f2);
  }
  else updatemaximum(b, fb);
}

void Maximizor::goldenstart(Double a1, Double a4, Double f1) {
  Double a2 = a1 + (a4 - a1)/(1 + GOLDENSECTION);
  Double f2 = f(a2, pos);
  if (f2 > f1) goldensearch(a1, a2, a4, f2);
  else updatemaximum(a1, f1);
}
