#include <math.h>
#include "map.h"
#include "files.h"
#include "montecarlo.h"
#include "vecutil.h"
#include "fmtout.h"
#include "family.h"
#include "probability.h"
#include "haldane.h"
#include "options.h"
#include "singlelocus.h"

#ifdef HAVE_ZLIB
#include "ExtData.hh"
#include "ExtFamily.hh"
#include "MarkerConfig.hh"

#include "ZUtils.hh"
#endif

Probability *Montecarlo::prob = 0;

extern Double runif();

Double choose(Uint n, Uint k) {
  Double nd = Double(n), kd = Double(k);
  return Uint(rint(exp(lgamma(nd + 1.0) - lgamma(nd - kd + 1.0) -
                       lgamma(kd + 1.0))));
}

IV bitbefore(IV v) {
  IV w = POW2[30];
  while ((w >> 1) > v) w >>= 1;
  return w;
}

Montecarlo *Montecarlo::getmontecarlo(Uint size, const string &haplo,
                                      const string &inher, const string &ihaplo,
                                      const string &founder,
                                      const string &markersfile) {
  string hap = (haplo == "" ? "mhaplo.out" : haplo);
  string inh = (inher == "" ? "minher.out" : inher);
  string iha = (ihaplo == "" ? "mihaplo.out" : ihaplo);
  string fou = (founder == "" ? "mfounder.out" : founder);
  for (Uint d = 0; d < distributions.size(); d++)
    optassert(distributions[d]->getpt() != "cpt",
              "More than one MONTECARLO lines.");
  return new Montecarlo(size, hap, inh, iha, fou, markersfile);
}

Montecarlo::Montecarlo(Uint size, const string &haplo, const string &inher,
                       const string &ihaplo, const string &founder,
                       const string &markersfile) :
    Haplodist(haplo, inher, ihaplo, founder, "cpt"), samplesize(size),
#ifdef HAVE_ZLIB
    data(0), zio(0),
#endif
    d(0) {
  if (markersfile != "") {
    Infile im;
    im.setname(markersfile);
    im.optcheck("Montecarlo markers");
    im.open();
    
    while (!im.eof() && !im.fail()) {
      markers.push_back("");
      im >> markers.back();
    }
  }
    
  simul = new Link[samplesize];
  setlinks();
}

void Montecarlo::setlinks() {
  for (Uint s = 0; s + 1 < samplesize; s++)
    simul[s].next = &simul[s + 1];
}

void Montecarlo::reset(Uint /*np*/) {
  if (d == 0) DELETEVEC(d);
  NEWVEC(IV, d, 16*1024*1024);
  simulated.resize(map->num);
  for (Uint s = 0; s < samplesize; s++) {
    DELETEVEC(simul[s].path);
    NEWVEC(IV, simul[s].path, map->num);
  }
  if (options->hapasshaplotypes || options->compresshaplotypes) {
    marker_idx.clear();
    if (markers.empty()) {
      if (!options->addendmarkers && map->num >= 1)
        marker_idx.push_back(0);
      for (Uint i = 1; i + 1 < map->num; i++)
        marker_idx.push_back(i);
      if (!options->addendmarkers && map->num >= 2)
        marker_idx.push_back(map->num - 1);
    }
    else {
      for (Uint m = 0; m < markers.size(); m++) {
        bool found = false;
        for (Uint i = 0; i < map->num && !found; i++)
          if (map->markername[i] == markers[m]) {
            marker_idx.push_back(i);
            found = true;
          }
      }
    }
    
    vector<string> markernames;
    for (unsigned int i = 0; i < marker_idx.size(); i++)
      markernames.push_back(map->markername[marker_idx[i]]);

#ifdef HAVE_ZLIB
    if (options->hapasshaplotypes) {
      delete data;
      data = new ExtData(markernames, "mc.out");
    } else {
      gzclose(zio);
      zio = gzopen(ihaplotypefile.name.c_str(), "wb");
      assertcond(zio, "Unable to open file " + ihaplotypefile.name +
                 " for writing");
      write(zio, markernames);
      write(zio, (unsigned int)samplesize);
    }
#endif
  }
  else {
    if (inherfile.is_open()) {
      inherfile.close();
      haplotypefile.close();
      ihaplotypefile.close();
      founderallelefile.close();
    }
    inherfile.open();
    haplotypefile.open();
    ihaplotypefile.open();
    founderallelefile.open();
    printheaders();
  }
}

Montecarlo::~Montecarlo() {
  delete [] simul;
#ifdef HAVE_ZLIB
  delete data;
  if (zio != 0) gzclose(zio);
#endif
  DELETEVEC(d);
}

void Montecarlo::nextfam(Uint pos, DoubleVec /*p0*/) {
  assertinternal(pos == 0);

//   cout << "Seed for " << curfamily()->id << " =\t" << options->seed[0]
//        << "\t" << options->seed[1] << "\t" << options->seed[2] << endl;
  
  lqreceived = false;
  // Calculate d
  Uint lastbeg = 0;
  d[0] = 0;
  IV v = 1;
  const Family *fam = curfamily();
  for (Uint k = 1; k <= fam->numbits; k++) {
    IV beg = v;
    for (IV i = lastbeg; i < beg; i++)
      for (IV e = bitbefore(d[i]); e < fam->numiv; e <<= 1)
        d[v++] = d[i] | e;
    lastbeg = beg;
  }
  assertcond(v == fam->numiv, "Bug in MC");

  for (Uint i = 0; i < simulated.size(); i++) simulated[i] = false;

  if (options->sexspecific) {
    malemask = femalemask = 0;
    for (Person *p = fam->firstdescendant; p != 0; p = p->next) {
      if (!p->father->founder()) malemask |= fam->mask & p->patmask;
      if (!p->mother->founder()) femalemask |= fam->mask & p->matmask;
    }
  }
  else
    malemask = femalemask = IV(-1);

#ifdef HAVE_ZLIB
  if (options->hapasshaplotypes) {
    assertinternal(data);
    data->addFamily();
  } else if (options->compresshaplotypes) {
    assertinternal(zio);
    write(zio, (unsigned int)fam->num);
    for (Person *p = fam->first; p != 0; p = p->next) {
      write(zio, p->id);
      if (p->founder()) {
        write(zio, "0");
        write(zio, "0");
      }
      else {
        write(zio, p->father->id);
        write(zio, p->mother->id);
      }
      write(zio, (unsigned int)p->sex);
      write(zio, (unsigned int)p->origdstat);
    }
  }
#endif
}

IV Montecarlo::getfcidx(IV v, Foundercouple *fc) {
  IV c = (v & fc->childrenmask)/fc->lastcbit()*(fc->mask/fc->lastgcbit + 1);
  IV w = (v & fc->mask)/fc->lastgcbit;
  return c | w;
}

void Montecarlo::Tfoundercouple(Float tht, FloatVec fctrans,
                                Foundercouple *fc) {
  Float n = wt[fc->mask];
  Uint i = 0;
  for (IV v = 0; v <= fc->childrenmask; v += fc->lastcbit())
    for (IV w = v; w <= v + fc->mask; w += fc->lastgcbit) {
      Float x = wt[fc->mask & w];
      Float Tchildren = (Tfounder(tht, fc->wife->mask, w)*
                         Tfounder(tht, fc->husband->mask, w));
      fctrans[i] = pow(tht, x)*pow(1.0 - tht, n - x)*Tchildren;
      i++;
    }
}

Float Montecarlo::Tfounder(Float tht, IV mask, IV v) {
  Float n = wt[mask] + 1;
  Float x = wt[mask & v];
  return pow(tht, x)*pow(1.0 - tht, n - x) + pow(tht, n - x)*pow(1.0 - tht, x);
}

Float Montecarlo::transition(IV v, Float tht, Float tht_female) {
  Float res = 1.0;
  for (Person *p = curfamily()->first; p != 0; p = p->next)
    if (p->fc == 0 && p->mask > 0) 
      res *= Tfounder((options->sexspecific && p->sex == FEMALE ?
                       tht_female : tht), p->mask, v);
  if (options->sexspecific) {
    Float n_male = wt[malemask];
    Float x_male = wt[malemask & v];
    Float n_female = wt[femalemask];
    Float x_female = wt[femalemask & v];
    res *= (pow(tht, x_male)*pow(1.0 - tht, n_male - x_male)*
            pow(tht_female, x_female)*
            pow(1.0 - tht_female, n_female - x_female));
  }
  else {
    Float n = wt[curfamily()->mask];
    Float x = wt[curfamily()->mask & v];
    res *= pow(tht, x)*pow(1.0 - tht, n - x);
  }
  return res;
}

#ifdef DEBUG_MC
Float nc[1000];
#endif // DEBUG_MC

void Montecarlo::set(FloatVec pv, Uint pos) {
  //
  //                          lq_t+1(v_t+1)
  // P(v_t | v_t+1, data) n_t -------------  =  T(v_t - v_t+1) lq_t(v_t)
  //                           q_t+1(v_t+1)
  //
  // Where n_t is the normalizing constant of lq_t(v_t), l may be
  // normalized by dividing mith l_nc stored in Probability class q
  // has been normalized

  // If this is the first set call then pv contains lq, otherwise it contains q
  if (!lqreceived) {
    lqreceived = true;
    lq = pv;
    return;
  }
  else
    lqreceived = false;
  
  Family *fam = curfamily();

  FloatVec q = pv;
  
  bool alldone = false; // True when all samples have been simulated
                        // at this marker
  int rm = prob->rightmarker[pos];
  int lm = prob->leftmarker[pos];
  Float tht = (lm != Probability::NOLEFTMARKER ? prob->theta[lm] : 0);
  Float tht_female = 0;
  if (options->sexspecific && lm != Probability::NOLEFTMARKER)
    tht_female = prob->theta_female[lm];
  simulated[lm] = true;
  Float n_t = 1.0/sum<Float>(lq, fam->numiv);
  setlinks();
  Link *first = simul;
  if (rm == Probability::NORIGHTMARKER) {
    // No informative markers right of this one so we simulate from the
    // marginal distribution at this marker
    Float cumsum = 0.0;
    for (Uint s = 0; s < samplesize; s++) simul[s].r = runif()/n_t;
    for (IV v = 0; v < fam->numiv && !alldone; v++) {
      if (q[v] == 0) continue;
      cumsum += lq[v];
      alldone = true;
      Link *last = 0;
      for (Link *s = first; s != 0;)
        if (cumsum >= s->r) {
          s->path[lm] = v;
          if (last != 0) s = last->next = s->next;
          else s = first = s->next;
        }
        else {
          alldone = false;
          last = s;
          s = s->next;
        }
    }
    assertinternal(cumsum*n_t < 1.0000001);
  }
  else {
    // Simulate given the next marker to the right
    for (Uint s = 0; s < samplesize; s++) {
      simul[s].r /= n_t;
      simul[s].cumsum = 0.0;
    }
    
    // Simulate v_t
    vector<FloatVec> Tfc(fam->numfc);
    Uint fcidx = 0;
    for (Foundercouple *fc = fam->firstfoundercouple;
         fc != 0; fc = fc->next, fcidx++) {
      NEWVEC(Float, Tfc[fcidx], POW2[wt[fc->mask | fc->childrenmask]]);
      Tfoundercouple(tht, Tfc[fcidx], fc);
    }
    for (Uint i = 0; i < fam->numiv && !alldone; i++) {
      Float T = transition(d[i], tht, tht_female);
#ifdef DEBUG_MC
      alldone = false;
#else
      alldone = true;
#endif // DEBUG_MC
      Link *last = 0;
      for (Link *s = first; s != 0;) {
        IV v_tplus1 = s->path[rm];
        IV v_t = v_tplus1 ^ d[i];
        if (q[v_t] == 0) {
          alldone = false;
          last = s;
          s = s->next;
          continue;
        } else
          s->path[lm] = v_t;
        Float Ts = T;
        Uint fcidx = 0;
        for (Foundercouple *fc = fam->firstfoundercouple;
             fc != 0; fc = fc->next, fcidx++) {
          IV piv_t = fc->pi(v_t) ^ fc->mask;
          IV piv_tplus1 = fc->pi(v_tplus1) ^ fc->mask;
          IV idx = getfcidx(d[i], fc);
          if (piv_t != v_t && piv_tplus1 != v_tplus1) {
            IV piidx = getfcidx(piv_t ^ v_tplus1, fc);
            Ts *= (1 - tht)*Tfc[fcidx][idx] + tht*Tfc[fcidx][piidx];
          }
          else
            Ts *= Tfc[fcidx][idx];
        }
        s->cumsum += Ts*lq[v_t];
        if (s->cumsum >= s->r) {
//          s->path[lm] = v_t;
          assertinternal(q[v_t] > 0);
#ifdef DEBUG_MC
          s->r = nc[0]/n_t + 3e300;
          s = s->next;
#else
          if (last != 0) s = last->next = s->next;
          else s = first = s->next;
#endif // DEBUG_MC
        }
        else {
          alldone = false;
          last = s;
          s = s->next;
        }
      }
    }
//    assertinternal(alldone);
    fcidx = 0;
    for (Foundercouple *fc = fam->firstfoundercouple;
         fc != 0; fc = fc->next, fcidx++) DELETEVEC(Tfc[fcidx]);
    
// #ifdef DEBUG_MC
//     if (fam->numbits > 0)
//       for (Uint s = 0; s < samplesize; s++)
//         assertinternal(fabs(simul[s].cumsum - nc[s]/n_t)/(nc[s]/n_t) < .01);
// #endif // DEBUG_MC
  }

  if (pos == 0 || prob->leftmarker[pos - 1] == Probability::NOLEFTMARKER) {
    // output results
    fillin();
    if (options->hapasshaplotypes) {
      vector<vector<unsigned int> > founder_config(samplesize);
      vector<vector<unsigned int> > affected_config(samplesize);

      for (Uint s = 0; s < samplesize; s++) {
        founder_config[s].resize(marker_idx.size());
        affected_config[s].resize(marker_idx.size());
      }
      
#ifdef HAVE_ZLIB
      for (Uint m = 0; m < marker_idx.size(); m++) {
        const Uint gam = marker_idx[m];
        MarkerConfig &config = data->back().addConfig();
        for (Uint s = 0; s < samplesize; s++)
          markerconfig(config, founder_config[s][m], affected_config[s][m], gam,
                       simul[s].path[gam]);
      }

      vector<string> founders;
      for (Person *p = fam->first; p != fam->firstdescendant; p = p->next)
        founders.push_back(p->id);

      data->back().write(fam->id, founders, false,
                         founder_config, affected_config);
#endif // HAVE_ZLIB
    }
    else
      for (Uint s = 0; s < samplesize; s++)
        ivout(simul[s].path, false);
#ifdef DEBUG_MC
    output(cout);
#endif // DEBUG_MC
  } else {
    // Calculate left side of equation above for the next round of simulations
    for (Uint s = 0; s < samplesize; s++) {
#ifdef DEBUG_MC
      nc[s] = lq[simul[s].path[lm]]/prob->l_nc[lm]/q[simul[s].path[lm]];
#endif // DEBUG_MC
      simul[s].r = runif()*
        lq[simul[s].path[lm]]/prob->l_nc[lm]/q[simul[s].path[lm]];
      assertinternal(finite(simul[s].r));
    }
  }
}

// Return recombination fraction between markers gam0 and gam1 using map
Float Montecarlo::recfrac(Uint gam0, Uint gam1, Uint type) {
  return recombfraccent(map->position[type][gam1] - map->position[type][gam0]);
}

void Montecarlo::fillin() {
  int firstmarker = -1;
  int lastmarker = -1;
  for (Uint gam = 0; gam < map->num; gam++) {
    if (simulated[gam] && firstmarker == -1) firstmarker = gam;
    if (simulated[gam]) lastmarker = gam;
  }
  assertinternal(firstmarker != -1);
  // Simulate from start of chromosome to first informative marker
  for (int gam = firstmarker - 1; gam >= 0; gam--)
    if (options->sexspecific)
      simulateend(gam + 1, gam, map->theta[1][gam], map->theta[2][gam]);
    else
      simulateend(gam + 1, gam, map->theta[0][gam]);
  // Simulate inbetween informative markers
  for (int gam = firstmarker + 1; gam < lastmarker; gam++)
    if (!simulated[gam]) {
      Uint rm = gam + 1;
      while (!simulated[rm]) rm++;
      if (options->sexspecific) 
        simulateinbetween(gam - 1, gam, rm, map->theta[1][gam - 1],
                          recfrac(gam, rm, 1), map->theta[2][gam - 1],
                          recfrac(gam, rm, 2));
      else
        simulateinbetween(gam - 1, gam, rm, map->theta[0][gam - 1],
                          recfrac(gam, rm, 0));
    }
  // Simulate from last informative marker to end of chromosome
  for (int gam = lastmarker + 1; gam < map->num; gam++)
    if (options->sexspecific)
      simulateend(gam - 1, gam, map->theta[1][gam - 1], map->theta[2][gam - 1]);
    else
      simulateend(gam - 1, gam, map->theta[0][gam - 1]);
}

#ifdef DEBUG_MC
void Montecarlo::output(ostream &out) {
  // Print results
  for (Uint gam = 0; gam < map->num; gam++) {
    cout << gam << "\t" << simulated[gam] << "\t";
    for (Uint s = 0; s < samplesize; s++)
      fmtout(out, 7, simul[s].path[gam]);
    out << "\n";
  }
}
#endif // DEBUG_MC

// Simulate v_1 | v_0
void Montecarlo::simulateend(IV gam0, IV gam1, Float tht, Float tht_female) {
  //
  // P(v_1 | v_0) =  T(v_1 - v_0)
  //
  assertinternal(!options->sexspecific || tht_female != -1);
  
  for (Uint s = 0; s < samplesize; s++) simul[s].path[gam1] = 0;
  
  // First simulate founder couple meiosis, then founder meiosis and
  // finally the rest
  setlinks();
  Family *fam = curfamily();
  vector<FloatVec> Tfc(fam->numfc);
  Uint fcidx = 0;
  for (Foundercouple *fc = fam->firstfoundercouple;
       fc != 0; fc = fc->next, fcidx++) {
    NEWVEC(Float, Tfc[fcidx], POW2[wt[fc->mask | fc->childrenmask]]);
    Tfoundercouple(tht, Tfc[fcidx], fc);
  }
  for (Link *l = simul; l != 0; l = l->next) {
    bool ldone = false;
    Uint fcidx = 0;
    for (Foundercouple *fc = fam->firstfoundercouple; fc != 0 && !ldone;
         fc = fc->next, fcidx++) {
      Float cumsum = 0.0;
      Float r = runif();
      for (IV v = 0; v < fc->wife->mask + fc->wife->lastbit && !ldone;
           v += fc->husband->lastbit)
        for (IV w = v; w <= v + fc->mask && !ldone; w += fc->lastgcbit) {
          Uint idx = getfcidx(w ^ l->path[gam0], fc);
          IV piidx = getfcidx(fc->pi(w) ^ l->path[gam0], fc);
          cumsum += ((1 - tht)*Tfc[fcidx][idx] + tht*Tfc[fcidx][piidx]);
          if (cumsum >= r) {
            l->path[gam1] |= (w ^ l->path[gam0]) &
              (fc->mask | fc->wife->mask | fc->husband->mask);
            ldone = true;
            break;
          }
        }
    }
  }
  fcidx = 0;
  for (Foundercouple *fc = fam->firstfoundercouple;
       fc != 0; fc = fc->next, fcidx++) DELETEVEC(Tfc[fcidx]);
   
  for (Link *l = simul; l != 0; l = l->next) {
    for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
      if (p->fc != 0) continue;
      bool ldone = false;
      Float cumsum = 0.0;
      Float r = runif();
      for (IV v = 0; v <= p->mask && !ldone; v += p->lastbit) {
        cumsum += Tfounder(options->sexspecific && p->sex == FEMALE ?
                           tht_female : tht, p->mask, v);
        if (cumsum >= r) {
          l->path[gam1] |= (v ^ l->path[gam0]) & p->mask;
          ldone = true;
        }
      }
    }
    
    for (IV v = 1; v <= fam->mask; v <<= 1) {
      Float t = tht;
      if (options->sexspecific && (v & femalemask)) t = tht_female;
      if (runif() < t) l->path[gam1] |= (v ^ l->path[gam0]) & fam->mask;
      else l->path[gam1] |= (v & l->path[gam0]) & fam->mask;
    }
  }
}

// Simulate v_1 | v_0 and v_2, when the marker order is 0, 1, 2
void Montecarlo::simulateinbetween(IV gam0, IV gam1, IV gam2,
                                   Float tht0, Float tht1,
                                   Float tht0_female, Float tht1_female) {
  //
  // P(v_1 | v_0, v_2) T(v_2 - v_0)  =  T(v_1 - v_0) T(v_2 - v_1)
  //
  assertinternal(!options->sexspecific ||
                 (tht0_female != -1 && tht1_female != -1));

  Float tht02 = tht0 + tht1 - 2.0*tht0*tht1;
  Float tht02_female = tht0_female + tht1_female - 2.0*tht0_female*tht1_female;
  for (Uint s = 0; s < samplesize; s++) simul[s].path[gam1] = 0;

  // First simulate founder couple meiosis, then founder meiosis and
  // finally the rest
  setlinks();
  Family *fam = curfamily();
  vector<FloatVec> Tfc0(fam->numfc);
  vector<FloatVec> Tfc1(fam->numfc);
  vector<FloatVec> Tfc2(fam->numfc);
  Uint fcidx = 0;
  for (Foundercouple *fc = fam->firstfoundercouple;
       fc != 0; fc = fc->next, fcidx++) {
    NEWVEC(Float, Tfc0[fcidx], POW2[wt[fc->mask | fc->childrenmask]]);
    NEWVEC(Float, Tfc1[fcidx], POW2[wt[fc->mask | fc->childrenmask]]);
    NEWVEC(Float, Tfc2[fcidx], POW2[wt[fc->mask | fc->childrenmask]]);
    Tfoundercouple(tht0, Tfc0[fcidx], fc);
    Tfoundercouple(tht1, Tfc1[fcidx], fc);
    Tfoundercouple(tht02, Tfc2[fcidx], fc);
  }
  for (Link *l = simul; l != 0; l = l->next) {
    bool ldone = false;
    Uint fcidx = 0;
    for (Foundercouple *fc = fam->firstfoundercouple; fc != 0 && !ldone;
         fc = fc->next, fcidx++) {
      Float cumsum = 0.0;
      Float r = runif()*
        ((1 - tht02)*Tfc2[fcidx][getfcidx(l->path[gam0] ^ l->path[gam2], fc)] +
         tht02*Tfc2[fcidx][getfcidx(fc->pi(l->path[gam0] ^ l->path[gam2]), fc)]);
      for (IV v = 0; v < fc->wife->mask + fc->wife->lastbit && !ldone;
           v += fc->husband->lastbit)
        for (IV w = v; w <= v + fc->mask && !ldone; w += fc->lastgcbit) {
          Uint idx0 = getfcidx(w ^ l->path[gam0], fc);
          IV piidx0 = getfcidx(fc->pi(w) ^ l->path[gam0], fc);
          Uint idx1 = getfcidx(w ^ l->path[gam2], fc);
          IV piidx1 = getfcidx(fc->pi(w) ^ l->path[gam2], fc);
          cumsum += (((1 - tht0)*Tfc0[fcidx][idx0] + tht0*Tfc0[fcidx][piidx0])*
                     ((1 - tht1)*Tfc1[fcidx][idx1] + tht1*Tfc1[fcidx][piidx1]));
          if (cumsum >= r) {
            l->path[gam1] |= w;
            ldone = true;
          }
        }
    }
  }
  fcidx = 0;
  for (Foundercouple *fc = fam->firstfoundercouple;
       fc != 0; fc = fc->next, fcidx++) {
    DELETEVEC(Tfc0[fcidx]);
    DELETEVEC(Tfc1[fcidx]);
    DELETEVEC(Tfc2[fcidx]);
  }
   
  for (Link *l = simul; l != 0; l = l->next) {
    for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
      Float t0 = tht0;
      Float t1 = tht1;
      Float t02 = tht02;
      if (options->sexspecific && p->sex == FEMALE) {
        t0 = tht0_female;
        t1 = tht1_female;
        t02 = tht02_female;
      }
      bool ldone = false;
      if (p->fc != 0) continue;
      Float cumsum = 0.0;
      Float r = runif()*Tfounder(t02, p->mask, l->path[gam0]^l->path[gam2]);
      for (IV v = 0; v <= p->mask && !ldone; v += p->lastbit) {
        cumsum += (Tfounder(t0, p->mask, v ^ l->path[gam0])*
                   Tfounder(t1, p->mask, v ^ l->path[gam2]));
        if (cumsum >= r) {
          l->path[gam1] |= v;
          ldone = true;
        }
      }
    }
    
    for (IV v = 1; v <= fam->mask; v <<= 1) {
      Float t0 = tht0;
      Float t1 = tht1;
      Float t02 = tht02;
      if (options->sexspecific && (v & femalemask)) {
        t0 = tht0_female;
        t1 = tht1_female;
        t02 = tht02_female;
      }
      Float T02 = Tbit(t02, l->path[gam0]^l->path[gam2], v);
      if (runif()*T02 < Tbit(t0, v ^ l->path[gam0], v)*
          Tbit(t1, v ^ l->path[gam2], v)) {
        l->path[gam1] |= v;
      }
    }
  }
}

void Montecarlo::printresults(const vector<IntMat> &curbits,
                              vector<IntMat> &alleles,
                              const vector<IntMat> founderalleles) {
  if (options->compresshaplotypes) {
#ifdef HAVE_ZLIB
    if (options->compresshaplotypes) {
      Family *fam = curfamily();
      for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
        for (Uint i = 0; i < marker_idx.size(); i++) {
          unsigned char a = alleles[marker_idx[i]][p->nmrk][0];
          write(zio, a);
        }
        for (Uint i = 0; i < marker_idx.size(); i++) {
          unsigned char a = alleles[marker_idx[i]][p->nmrk][1];
          write(zio, a);
        }
      }
    }
#endif // HAVE_ZLIB
  }
  else
    Haplodist::printresults(curbits, alleles, founderalleles);
}

#ifdef HAVE_ZLIB
void Montecarlo::markerconfig(MarkerConfig &config, unsigned int &fc,
                              unsigned int &ac, Uint gam, IV v) {
  graph->haploreset();
  Family *fam = curfamily();
  for (Person *p = fam->firstdescendant; p != 0; p = p->next) {
    if (options->sexlinked && p->sex == MALE)
      p->nod[0] = p->nod[1] = p->mother->nod[p->matmask & v ? 1 : 0];
    else {
      p->nod[0] = p->father->nod[p->patmask & v ? 1 : 0];
      p->nod[1] = p->mother->nod[p->matmask & v ? 1 : 0];
    }
  }
  for (Person *p = fam->first; p != 0; p = p->next)
    if (p->gen[0] != 0 && p->genotyp[gam])
      assertcond(graph->addgenotype(p->nod[0], p->nod[1],
                                    p->gen[0][gam], p->gen[1][gam]),
                 "No consistent allele assignment at " +
                 map->markername[gam] + " in Monte Carlo simulation");

  vector<unsigned int> founders(2*fam->numf, ALLELEUNKNOWN);
  vector<unsigned int> affecteds(2*fam->numf, 0);

  for (Person *p = fam->first; p != 0; p = p->next)
    if (p->origdstat == AFFECTED) {
      affecteds[p->nod[0]->index]++;
      affecteds[p->nod[1]->index]++;
    }

  const Allele BLOCK = Allele(-1);
  
  for (Node *n = graph->first; n != 0; n = n->next) {
    if (n->allele == ALLELEUNKNOWN && n->edge != 0) {
      vector<unsigned int> b1, b2;
      n->getBlock(b1, b2);

      Uint ba = config.addBlock(b1, b2, n->edge->a0, n->edge->a1);

      for (Uint i = 0; i < b1.size(); i++)
        founders[b1[i]] = ba;
      for (Uint i = 0; i < b2.size(); i++)
        founders[b2[i]] = ba;
    }
    else if (n->allele != ALLELEUNKNOWN && n->allele != BLOCK)
      founders[n->index] = n->allele;
  }

  fc = config.addFoundersAssignment(founders);
  ac = config.addAffectedsAssignment(affecteds);
}
#endif
