#include <time.h>
#include <unistd.h>
#include "files.h"
#include "options.h"
#include "warning.h"
#include "model.h"
#include "asmmodel.h"
#include "asmpoly.h"
#include "asmexp.h"
#include "linkagedist.h"
#include "paramdist.h"
#include "markermodel.h"
#include "calcqtl.h"
#include "qtlmodel.h"
#include "recombmodel.h"
#include "linkagemodel.h"
#include "parmodel.h"
#include "ibdpairmodel.h"
#include "condlikmodel.h"
#include "varcompmodel.h"
#include "assocmodel.h"
#include "simulation.h"
#include "entropy.h"
#include "parseline.h"
#include "modelweight.h"
#include "calc.h"
#include "calcscore.h"
#include "haplomodel.h"
#include "forcedmodel.h"
#include "montecarlo.h"
#include "founderhaplomodel.h"

const int DEFAULTSTEPS = 2;

Options *options;

//////////////////////////////////////////////////////////////////////
// Utility functions

int bool2int(bool a) {return a ? 1 : 0;}

Double readnumericoption(const string &s, const string &t) {
  Uint len = t.length();
  string a = s.substr(0, len);
  lowercase(a);
  if (a == t)
    return atof(s.substr(len, s.length() - len).c_str());
  else return -1.0;
}

Double readpenetrance(string &p) {
  string::size_type idx = p.find_first_of('/');
  optassert(idx != string::npos, "Missing '/' in penetrance clause");
  Double pe = atof(p.substr(0, idx).c_str());
  optassert(pe >= 0.0 && pe <= 1.0, string("Illegal value (") + pe +
            ") for penetrance");
  p = p.substr(idx + 1, p.length() - idx - 1);
  return pe;
}
  
void blankcomment (string& s) {
  Uint i = s.find('%');
  if (i != string::npos) s.erase(i, s.length());
}

//////////////////////////////////////////////////////////////////////
// Keyword table for parsing options file

template<class T>  // A keyword-pointer pair for use in list
class Kwdptr {
public:
  Kwdptr(const string &k, T *p, Kwdptr<T> *n) : kwd(k), ptr(p), next(n) {}
  ~Kwdptr() {
    delete next;
  }
  string kwd;
  T *ptr;
  Kwdptr *next;
};

template<class T>  // A list of keyword-pointer pairs
class Table {
public:
  Table() : first(0) {}
  ~Table() {
    delete first;
  }
  void add(const string& s, T *f) {
    first = new Kwdptr<T>(s, f, first);
  }
  bool tryout(const string &keyw, ISTRINGSTREAM &ss) {
    // Return true if keyw found in keyword list and in this case the value of
    // keyw is read from the stream is into the variable pointed to by the
    // pointer corresponding to keyword.
    T *p = find(keyw);
    if (p) get(p, keyw, ss);
    return p != 0;
  }
  T *find(const string& s) { // Find s in keword list, return corresponding ptr or 0.
    Kwdptr<T> *kf = first;
    while (kf && s != kf->kwd)
      kf = kf->next;
    return kf ? kf->ptr : 0;
  }
  void get(T *p, const string& s, ISTRINGSTREAM& ss); // Get value from ss
  Kwdptr<T> *first;
};

template<>
void Table<bool>::get(bool *bp, const string& /*keyw*/, ISTRINGSTREAM& ss) {
  string val;
  ss >> val >> ws;
  lowercase(val);
  bool ok = ss.eof() && (val == "on" || val == "off");
  *bp = !ok || val != "off";
  if (!ok) optwarning("Illegal value - assuming 'on'");
}

template<>
void Table<File>::get(File *fp, const string& keyw, ISTRINGSTREAM& ss) {
  string name;
  ss >> name >> ws;
  fp->setname(name);
  fp->optcheck(keyw);
}

template<class T>
void Table<T>::get(T *ip, const string& keyw, ISTRINGSTREAM& ss) {
  *ip = -1;
  ss >> *ip >> ws;
  bool ok = (ss.good() || ss.eof()) && *ip >= 0;
  if (!ok) {
    optwarning("Illegal value of " + keyw + " - set to 0");
    *ip = 0;
  }
}


//////////////////////////////////////////////////////////////////////
// Options class function definitions
    
void Options::prefileline(ISTRINGSTREAM& ss) {
  string pre;
  while (!ss.eof()) {
    ss >> pre >> ws;
    Infile ifile;
    ifile.setname(pre);
    ifile.optcheck("prefile");
    prefile.push_back(pre);
  }
}

void Options::datfileline(ISTRINGSTREAM& ss) {
  string dat;
  while (!ss.eof()) {
    ss >> dat >> ws;
    Infile ifile;
    ifile.setname(dat);
    ifile.optcheck("datfile");
    datfile.push_back(dat);
  }
}

void Options::nullfileline(ISTRINGSTREAM& ss) {
  string sc, filename;
  ss >> sc >> filename >> ws;
  if (sc != "")
    Distribution::setnullfilename(sc, filename);
  else
    optwarning("Illegal nullfile option, use e.g. 'NULLFILENAME pairs'");
}

void Options::probfileline(ISTRINGSTREAM& ss) {
  string sc, filename, pt;
  ss >> pt >> sc >> filename >> ws;
  if (sc != "")
    Distribution::setprobfilename(pt, sc, filename);
  else
    optwarning("Illegal probfile option, use e.g. 'PROBFILENAME mpt pairs'");
}

void Options::chromosomesline(ISTRINGSTREAM& ss) {
  optassert(chromolist == 0, "Only one CHROMOSOMES option allowed");
  Chromolist *last = 0;
  while (!ss.eof() && ss.good()) {
    string ch;
    ss >> ch >> ws;
    string lch = ch;
    lowercase(lch);
    if (last == 0 && lch == "all") {
      chromolist = new Chromolist("all");
      return;
    }
    optassert(lch != "all", "all used with other chromosomenames in CHROMOSOMES");
    if (last == 0) chromolist = last = new Chromolist(ch);
    else {
      last->next = new Chromolist(ch);
      last = last->next;
    }
  }
  optassert(chromolist != 0,
            "At least one chromosome (or all) must be specified in CHROMOSOMES option");
}

void Options::pairwiseibdline(ISTRINGSTREAM& ss) {
  string pt, type, prior, posterior;
  ss >> pt >> type >> prior >> posterior;
  optassert(type == "all" || type == "genotyped" || type == "affected" ||
            type == "qtl" || type == "informative", string("Illegal type of ") +
            "pairwise ibd, use all, genotyped, affected, qtl or informative " +
            "instead");
  new IBDpairmodel(pt, type, prior, posterior);
  calcspt = true;
  calcmpt |= pt != "spt";
}

void Options::famfilesline(ISTRINGSTREAM &ss) {
  optassert(famfiles == OB_NOTSET, "Famfiles flag set twice in option file");
  string opt;
  ss >> opt;
  lowercase(opt);
  if (opt == "" || opt == "on" || opt == "yes") famfiles = OB_ON;
  else if (opt == "off" || opt == "no") famfiles = OB_OFF;
  else optassert(false, "Famfiles flag not regocnised '" + opt + "'");
}

void Options::printoriginalallelesline(ISTRINGSTREAM& ss) {
  string unkn, shift;
  ss >> unkn >> shift >> ws;
  if (unkn != "") unknownrepeatstring = unkn;
  if (shift != "") shiftrepeats = atoi(shift.c_str());
  printoriginalalleles = true;
}


void Options::modelline(ISTRINGSTREAM& ss) {
  ParseItems items;

  ParseItems npt;
  ParseKeyword mpt("mpt", npt);
  ParseKeyword spt("spt", npt);
  ParseKeyword dpt("dpt", npt);
  items.insert(items.end(), npt.begin(), npt.end());

  ParseItems modeltype;
  ParseKeyword exp("exp", modeltype);
  ParseKeyword lin("lin", modeltype);
  ParseNumeric poly("poly", modeltype, 0, 1, 100);
  ParseKeyword par("par", modeltype);
  ParseKeyword varcomp("varcomp", modeltype);
  ParseKeyword condlik("condlik", modeltype);
  ParseKeywordArgument marker("marker", modeltype);
  ParseKeyword assoc("assoc", modeltype);
  items.insert(items.end(), modeltype.begin(), modeltype.end());

  ParseItems mode;
  ParseKeyword additive("additive", mode);
  ParseKeyword dominant("dominant", mode);
  items.insert(items.end(), mode.begin(), mode.end());
  
  ParseItems scorefuncs;
  ParseItems simplescorefuncs;
  ParseKeyword pairs("pairs", simplescorefuncs);
  ParseKeyword all("all", simplescorefuncs);
  ParseKeyword homoz("homoz", simplescorefuncs);
  ParseKeyword robdom("robdom", simplescorefuncs);
  ParseKeyword mnallele("mnallele", simplescorefuncs);
  scorefuncs.insert(scorefuncs.end(), simplescorefuncs.begin(),
                    simplescorefuncs.end());
  ParseNumericVector ps("ps", scorefuncs, 0, 1, 3, 3);
  ParseItems qtlcorrfuncs;
  ParseKeyword qtlhe("qtlhe", qtlcorrfuncs);
  ParseKeyword qtlnhe("qtlnhe", qtlcorrfuncs);
  ParseKeyword qtlscore("qtlscore", qtlcorrfuncs);
  ParseItems qtlvcfuncs;
  ParseKeyword qtlpairs("qtlpairs", qtlvcfuncs);
  ParseKeyword qtlpairsncp("qtlpairsncp", qtlvcfuncs);
  ParseKeyword qtlgenpairsncp("qtlgenpairsncp", qtlvcfuncs);
  ParseKeyword qtlfamily("qtlfamily", qtlvcfuncs);
  ParseKeyword qtlfamilyncp("qtlfamilyncp", qtlvcfuncs);
  ParseKeyword qtlwpc("qtlwpc", scorefuncs);
  ParseKeywordArgument genpairs("genpairs", scorefuncs);
  scorefuncs.insert(scorefuncs.end(), qtlcorrfuncs.begin(),
                    qtlcorrfuncs.end());
  scorefuncs.insert(scorefuncs.end(), qtlvcfuncs.begin(), qtlvcfuncs.end());
  items.insert(items.end(), scorefuncs.begin(), scorefuncs.end());

  ParseItems weighting;
  ParseKeyword equal("equal", weighting);
  ParseNumeric power("power", weighting, 0, -100000, 100000);
  ParseKeyword hodge("hodge", weighting);
  ParseKeywordArgument weights("weights", weighting);
  items.insert(items.end(), weighting.begin(), weighting.end());

  ParseNumeric waff("waff", items, 0, -10000, 10000);
  ParseNumeric wunaff("wunaff", items, 0, -10000, 10000);
  ParseKeyword datfreq("datfreq", items);
  ParseKeywordArgument patientfile("patientfile", items);
  ParseKeywordArgument controlfile("controlfile", items);
  
  ParseKeyword het("het", items);
  ParseKeyword X("X", items);
  ParseNumeric freq("freq", items, -1, 0, 1);
  ParseNumericVector pen("pen", items, 0, 1, 3, 5);

  ParseNumeric shared("shared", items, 0, 0, 1);
  ParseNumeric corr("corr", items, 0, -100000, 100000);
  ParseNumeric var("var", items, 1, -100000, 100000);
  ParseNumericVector vc("vc", items, 0, 1, 3, 4);
  ParseKeyword pihat("pihat", items);

  ParseKeyword nolinkage("nolinkage", items);
  
  Stringvector remainder;
  parseline(items, ss, remainder);

  checkexactlyone(npt, "MODEL");
  string pt;
  if (mpt()) pt = "mpt";
  else if (spt()) pt = "spt";
  else if (dpt()) pt = "dpt";
  
  checkexactlyone(modeltype, "MODEL");

  string outfile = "";
  string foutfile = "";
  if (remainder.size() > 0) {
    outfile = remainder[0];
    if (remainder.size() > 1) {
      foutfile = remainder[1];
      optassert(remainder.size() <= 2,
                "Too many filenames, or unknown options given in MODEL line");
    }
  }
  
  if (exp() || lin() || poly()) {
    checkexactlyone(scorefuncs, "MODEL");
    Calcscore *calc = 0;
    for (Uint sc = 0; sc < simplescorefuncs.size(); sc++)
      if ((*simplescorefuncs[sc])())
        calc = Calcqualitativescore::getcalcqualitativescore(
          simplescorefuncs[sc]->getkeyword());
    if (ps())
      calc = Calcparentspecificscore::getcalcparentspecificscore(ps.getkeyword(),
                                                                 ps.getvalue(0),
                                                                 ps.getvalue(1),
                                                                 ps.getvalue(2));
    for (Uint sc = 0; sc < qtlcorrfuncs.size(); sc++)
      if ((*qtlcorrfuncs[sc])()) {
        traitfilerequired = true;
        calc = Calcgenpairsshared::getcalcgenpairsshared(qtlcorrfuncs[sc]->
                                                         getkeyword(),
                                                         (corr() ?
                                                          corr.getvalue() : 0),
                                                         (var() ?
                                                          var.getvalue() : 1));
      }
    for (Uint sc = 0; sc < qtlvcfuncs.size(); sc++) {
      if ((*qtlvcfuncs[sc])()) {
        traitfilerequired = true;
        optassert(vc(), "vc: option must be used with the " +
                  qtlvcfuncs[sc]->getkeyword() + " scoring function");
        Double s2, s, a, d;
        if (vc.numvalues() == 3) {
          optassert(vc.getvalue(0) + vc.getvalue(1) + vc.getvalue(2) <= 1.0,
                    "Sum of variance components is greater than 1");
          s = vc.getvalue(0);
          a = vc.getvalue(1);
          d = vc.getvalue(2);
          s2 = 1 - (s + a + d);
        }
        else {
          s2 = vc.getvalue(0);
          s = vc.getvalue(1);
          a = vc.getvalue(2);
          d = vc.getvalue(3);
        }
        if (qtlfamily() || qtlfamilyncp())
          calc = CalcQTL::getcalcQTL(qtlvcfuncs[sc]->getkeyword(),
                                     s2, s, a, d);
        else
          calc = Calcgenpairsvc::getcalcgenpairsvc(qtlvcfuncs[sc]->getkeyword(),
                                                   s2, s, a, d);
      }
    }
    if (qtlwpc()) {
      traitfilerequired = true;
      calc = Calcgenpairsshared::getcalcgenpairsshared(qtlwpc.getkeyword(),
                                                       0, 1);
    }
    else if (genpairs())
      calc = Calcgenpairsfile::getcalcgenpairsfile(genpairs.getkeyword(),
                                                   genpairs.getargument());
    optassert(calc != 0, "Internal error: scoring function not handled");

    ASMmodel *asmmodel = 0;
    if (exp()) asmmodel = new ASMexp(calc, pt);
    else if (lin()) asmmodel = new ASMpoly(calc, pt, "lin", 1);
    else if (poly()) {
      Double rr = poly.getvalue();
      asmmodel = new ASMpoly(calc, pt, string("poly:") + Uint(poly.getvalue()),
                             Uint(rr));
    }

    /*****************************/
    checkexactlyone(weighting, "MODEL");
    if (equal()) asmmodel->setweights(new Modelweightequal);
    else if (power())
      asmmodel->setweights(new Modelweightpower(power.getvalue()));
    else if (hodge()) asmmodel->setweights(new Modelweighthodge);
    else if (weights())
      asmmodel->setweights(new Modelweightfile(weights.getargument()));
    asmmodel->setupfiles(outfile, foutfile);
  }
  else if (par()) {
    optassert(freq() && pen() || !freq() && !pen(), string("Either specify ") +
              "both frequendy and penetrance in MODEL line, or neither");
    if (X() && freq())
      new Parmodel(pt, het(), outfile, foutfile, freq.getvalue(),
                   pen.getvalue(0), pen.getvalue(1), pen.getvalue(2),
                   pen.getvalue(3), pen.getvalue(4));
    else if (!X() && freq())
      new Parmodel(pt, het(), outfile, foutfile,freq.getvalue(),
                   pen.getvalue(0), pen.getvalue(1), pen.getvalue(2)); 
    else new Parmodel(pt, het(), outfile, foutfile);
  }
  else if (varcomp() || condlik()) {
    traitfilerequired = true;
    optassert(itemscount(mode) > 0 || !shared(),
              "Either additive or dominant must be given or shared variance not fixed");
    if (varcomp())
       new Varcompmodel(pt, !shared(), additive(), dominant(),
                       (shared() ? shared.getvalue() : -1), pihat(),
                       outfile, foutfile);
    else if (condlik())
      new Condlikmodel(pt, !shared(), additive(), dominant(),
                       (shared() ? shared.getvalue() : -1), 
                       outfile, foutfile);
  }
  else if (marker()) {
    new Markermodel(marker.getargument(), X(), pt, outfile, foutfile);
  }
  else if (assoc()) {
    if (weights()) 
      new Assocmodel(pt, weights.getargument(), !datfreq(),
                     controlfile.getargument(), patientfile.getargument(),
                     outfile, foutfile);
    else
      new Assocmodel(pt, waff.getvalue(), wunaff.getvalue(), !datfreq(),
                     controlfile.getargument(), patientfile.getargument(),
                     outfile, foutfile);
  }
  else assertinternal(false);

  calcspt = true;
  calcmpt |= mpt() || dpt();
  calcdpt |= dpt();
}

void Options::xoverline(ISTRINGSTREAM& ss) {
  string crossoverfile, fcrossoverfile;
  ss >> crossoverfile >> fcrossoverfile >> ws;
  new Recombmodel(crossoverfile, fcrossoverfile);
  calcspt = calcrpt = calcmpt = true;
}

void Options::uninformativeline(ISTRINGSTREAM& ss) {
  string uninf;
  ss >> uninf >> ws;
  uninformative.setname(uninf,"uninformative.out");
  uninformative.optcheck("uninformative marker file");
}

void Options::allowinconsistentline(ISTRINGSTREAM& ss) {
  string incons;
  ss >> incons >> ws;
  inconsistent.setname(incons,"inconsistent.out");
  inconsistent.optcheck("inconsistent marker file");
}

void Options::haplotypeline(ISTRINGSTREAM& ss) {
  dohaplo = true;
  calcspt = true;
  string haplo, ihaplo, founder, inher;
  ss >> haplo >> ihaplo >> founder >> inher;
  new Haplomodel(haplo, ihaplo, founder, inher);
}

void Options::montecarloline(ISTRINGSTREAM& ss) {
  calcspt = true;
  calcmpt = true;
  montecarlo = true;

  ParseItems items;
  ParseKeywordArgument markers("markers", items);
  
  Stringvector remainder;
  parseline(items, ss, remainder);

  Uint n = remainder.empty() ? 0 : atoi(remainder[0].c_str());
  if (n == 0) {
    n = 1;
    optwarning("Only one iteration will be performed in Monte Carlo");
  }
  while (remainder.size() < 5) remainder.push_back("");
  
  Montecarlo::getmontecarlo(n, remainder[0], remainder[1], remainder[2],
                            remainder[3], markers.getargument());
}

void Options::foundercountline(ISTRINGSTREAM& ss) {
  string of, b;
  ss >> of >> b;
  bool ordered = false;
  if (of == "ordered") {
    ordered = true;
    of = b;
  }
  new Founderhaplomodel(of, ordered);
}

void Options::seedline(ISTRINGSTREAM& ss) {
  Uint a;
  ss >> a >> ws;
  if (ss.eof()) {
    seed[0] = a >> 16;
    seed[1] = a % (unsigned short int)(-1);
    seed[2] = 0x330E;
  }
  else {
    seed[0] = a;
    ss >> seed[1] >> seed[2] >> ws;
  }
}

void Options::forcedxoversline(ISTRINGSTREAM& ss) {
  doforcedrecomb = true;
  calcspt = true;
  string forcedrecomb, famforcedrecomb;
  ss >> forcedrecomb >> famforcedrecomb;
  new Forcedmodel(forcedrecomb, famforcedrecomb);
}

void Options::simulateline(ISTRINGSTREAM& ss) {
  optassert(simulation == 0, "Only one simulation can be performed each run");
  simulation = new Simulation();

  ParseItems items;
  ParseNumeric dloc("dloc", items, -1, -100000, 100000);
  ParseNumeric npre("npre", items, 1, 1, UINT_MAX);
  ParseNumeric rep("rep", items, 1, 1, UINT_MAX);
  ParseNumeric err("err", items, 0, 0, 1);
  ParseNumeric yield("yield", items, 1, 0, 1);
  ParseNumeric het("het", items, 0, 0, 1);
  ParseKeyword perfectdata("perfectdata", items);
  ParseNumericVector interf("interf", items, 0, 1000000, 2, 2);

  Stringvector remainder;
  parseline(items, ss, remainder);
  ss >> ws;
  if (!ss.eof()) {
    string rem;
    ss >> rem;
    cerr << rem << endl;
  }
  if (remainder.size() > 0)
    optassert(false, "Illegal option in SIMULATE: " + remainder[0]);
  optassert(!perfectdata() || err.getvalue() == 0.0 && yield.getvalue() == 1.0,
            "Can not simulate errors/yield and perfect data at the same time");

  simulation->diseasepos = dloc.getvalue();
  simulation->nsimulations = Uint(npre.getvalue());
  simulation->nrepeats = Uint(rep.getvalue());
  simulation->errorrate = err.getvalue();
  simulation->yield = yield.getvalue();
  simulation->hetrogen = het.getvalue();
  simulation->perfectdata = perfectdata();
  if (interf()) {
    simulation->nu_male = interf.getvalue(0);
    simulation->nu_female = interf.getvalue(1);
  }
}

Options::Options(const string& optionsfile) :
    checkcondition(35),
    chromolist(0),
    controlprior(2.0),
    addendmarkers(true),
    printpairscores(false),
    steps(DEFAULTSTEPS),
    maxlocusforoutput(3),
    maxsteplength(.0),
    readprobfiles(false),
    writeprobfiles(false),
    calcspt(false),
    calcmpt(false),
    calcdpt(false),
    calcrpt(false),
    dohaplo(false),
    doforcedrecomb(false),
    foundercouples(true),
    nplexactp(false),
    lodexactp(false),
    entropy(false),
    sexspecific(false),
    nplpfile(false),
    sexlinked(false),
    markerdistzero(false),
    montecarlo(false),
    haplotypefixbits(false),
    compresshaplotypes(false),
    hapasshaplotypes(false),
    warnonmultiplemaxima(false),
    printoriginalalleles(false),
    pseudoautosomal(false),
    famfiles(OB_NOTSET),
    faminfo(false),
    unknownrepeat(-999999),
    unknownrepeatstring(""),
    shiftrepeats(0),
    xoverrateprecision(3),
    allownegativevc(false),
    nosplitting(false),
    unit(DEFAULT),
    simulation(0),
    swapdirname(""),
    swap(false),
    inherdistfolder("inhervecs"),
    estimatememory(false),
    traitfilerequired(false) {
  time_t curtime;
  time(&curtime);
  seed[0] = curtime >> 16;
  seed[1] = curtime % (unsigned short int)(-1);
  seed[2] = 0x330E;

  ifstream optfil(optionsfile.c_str());
  if (optfil == 0) fatal("Cannot open optionsfile " + optionsfile);
  message("Using options file " + optionsfile);

  Table<File> ftbl;
  ftbl.add("genotypefile", &genotypefile);
  ftbl.add("mapfile", &mapfile);
  ftbl.add("chromosomefile", &chromosomefile);
  ftbl.add("freqfile", &freqfile);
  ftbl.add("affectionfile", &affectionfile);
  ftbl.add("oldfreqfile", &oldfreqfile);
  ftbl.add("traitfile", &traitvaluefile);
  ftbl.add("stepfile", &stepfile);
  ftbl.add("lfile", &lfile);
  ftbl.add("rfile", &rfile);
  ftbl.add("qfile", &qfile);
  ftbl.add("pfile", &pfile);
  ftbl.add("calclikelihood", &likelihoodfile);
  ftbl.add("deltabounds", &deltaboundfile);

  Table<Uint> itbl;
  itbl.add("steps", &steps);
  itbl.add("maxlocusforoutput", &maxlocusforoutput);
  itbl.add("maxmemory", &maxmem);
  itbl.add("maxunloop", &maxunloop);
  itbl.add("checkcondition", &checkcondition);
  itbl.add("xoverrateprecision", &xoverrateprecision);

  Table<Float> dtbl;
  dtbl.add("maxsteplength", &maxsteplength);
  dtbl.add("controlprior", &controlprior);

  Table<bool> btbl;
  btbl.add("readprobfiles", &readprobfiles);
  btbl.add("writeprobfiles", &writeprobfiles);
  btbl.add("foundercouples", &foundercouples);
  btbl.add("nplexactp", &nplexactp);
  btbl.add("lodexactp", &lodexactp);
  btbl.add("entropy", &entropy);
  btbl.add("nplpfile", &nplpfile);
  btbl.add("markerdistzero", &markerdistzero);
  btbl.add("sexspecific", &sexspecific);
  btbl.add("warnonmultiplemaxima", &warnonmultiplemaxima);
  btbl.add("haplotypefixbits", &haplotypefixbits);
  btbl.add("hapasshaplotypes", &hapasshaplotypes);
  btbl.add("compresshaplotypes", &compresshaplotypes);
  btbl.add("addendmarkers", &addendmarkers);
  btbl.add("printpairscores", &printpairscores);
  btbl.add("pseudoautosomal", &pseudoautosomal);
  btbl.add("allownegativevc", &allownegativevc);
  btbl.add("nosplitting", &nosplitting);
  btbl.add("faminfo", &faminfo);

  string lin, keyword, value;
  int linnum = 0;

  while (!optfil.eof()) {
    linnum++;
    setoptlinenum(linnum);
    getline(optfil, lin);
    blankcomment(lin);
#ifdef HAVE_SSTREAM
    ISTRINGSTREAM ss(lin);
#else
    ISTRINGSTREAM ss(lin.c_str());
#endif
    keyword = "";
    ss >> keyword >> ws;
    lowercase(keyword);
    if (ftbl.tryout(keyword, ss)); // simple file keyword
    else if (itbl.tryout(keyword, ss)); // integer keyword
    else if (dtbl.tryout(keyword, ss)); // double keyword
    else if (btbl.tryout(keyword, ss)); // bool keyword
    else if (keyword == "unit") {
      ss >> value >> ws;
      lowercase(value);
      if (value == "centimorgan" || value == "centimorgans")
        unit = CENTIMORGAN;
      else if (value == "recombination"  || value == "recombinations")
        unit = RECOMBINATION;
      else
        optwarning("Illegal unit, using default");
    }
    else if (keyword == "prefile")
      prefileline(ss);
    else if (keyword == "datfile") 
      datfileline(ss);
    else if (keyword == "crossoverrate" || keyword == "crossoverfile")
      xoverline(ss);
    else if (keyword == "uninformative" || keyword == "uninformativefile")
      uninformativeline(ss);
    else if (keyword == "allowinconsistent")
      allowinconsistentline(ss);
    else if (keyword == "model")
      modelline(ss);
    else if (keyword == "nullfilename")
      nullfileline(ss);
    else if (keyword == "probfilename")
      probfileline(ss);
    else if (keyword == "simulate")
      simulateline(ss);
    else if (keyword == "haplotype")
      haplotypeline(ss);
    else if (keyword == "montecarlo")
      montecarloline(ss);
    else if (keyword == "forcedxovers")
      forcedxoversline(ss);
    else if (keyword == "inherdistfolder")
      ss >> inherdistfolder;
    else if (keyword == "swapdirname")
      ss >> swapdirname;
    else if (keyword == "chromosomes")
      chromosomesline(ss);
    else if (keyword == "printoriginalalleles")
      printoriginalallelesline(ss);
    else if (keyword == "pairwiseibd")
      pairwiseibdline(ss);
    else if (keyword == "famfiles")
      famfilesline(ss);
    else if (keyword == "foundercount")
      foundercountline(ss);
    else if (keyword == "seed")
      seedline(ss);
    else if (keyword != "") {
      optwarning("Unknown option");
      char buf[500];
      ss.getline(buf, 500);
    }
    if (!ss.eof()) optwarning("Extra characters at end of line");
  }

  assertcond(!readprobfiles,
             "The readprobfiles feature has been temporarily turned off");
  
  if (entropy) {
    if (calcmpt) Entropy::getentropy("mpt");
    Entropy::getentropy("spt");
  }

  optassert(steps > 0, "Minimum number of steps is 1.");
  
  bool linkageinput = prefile.size() > 0 && datfile.size() > 0;
  bool decodeinput = (genotypefile.assigned() && mapfile.assigned() &&
                      freqfile.assigned() && chromosomefile.assigned());
  optassert(simulation == 0 || Model::modelcount() == 0,
            "Simulation and analysis can not be performed in the same run");
  optassert(readprobfiles || linkageinput || decodeinput
            , string("Neither linkage inputfiles (pre/dat)") +
            " or probfiles given for input");
  optassert(simulation == 0 || linkageinput,
            string("Both a prefile and a datfile must be given for") +
            " simulation to be performed");
  optassert(chromolist != 0 || !decodeinput, 
            "CHROMOSOMES option must be specified with Decode format");
  optassert(!affectionfile.assigned() || decodeinput,
            "AFFECTINFILE option must only be specified with Decode format");
  optassert(!linkageinput || prefile.size() == datfile.size(),
            "The same number of DATFILEs and PREFILEs must be supplied");
  optassert(!decodeinput || prefile.size() == 1,
            "Only one prefile may be specified with Decode format");
  optassert(!printoriginalalleles || decodeinput,
            "Can not use PRINTORIGINALALLELES with old style input");
  optassert(!traitfilerequired || traitvaluefile.name != "",
            "Traitfile must be given when running QTL analysis");
  optassert(!compresshaplotypes || !hapasshaplotypes,
            "Not both COMPRESSHAPLOTYPES and HAPASSHAPLOTYPES may be on");
  if (likelihoodfile.assigned()) calcspt = calcmpt = true;
  if (swapdirname == "") swapdirname = ".";
  swapdirname = swapdirname + "/" + Uint(getpid());
  if (readprobfiles || writeprobfiles) {
    // If nullfiles & probfiles not specified, set to default
    Distribution::setnames(readprobfiles, writeprobfiles);
  }
  // Find out if q and p calculation should be skipped
  skipfirstphase = Distribution::probfilesexist();
  optassert(linkageinput || decodeinput || readprobfiles && skipfirstphase,
            string("Not all probs and nullprobs are available for reading") +
            "(or trying to run a parametric model without pre/dat-files)");
  if (maxmem == 0) maxmem = 1024*1024*2047;
  else maxmem *= 1024*1024;
  if (maxunloop == 0) maxunloop = 4;
  else if (maxunloop > 4) fatal("Illegal maximum number of bits to unloop");

  if (!decodeinput) addendmarkers = false;
  
  message("");
  if (linkageinput) {
    message("Using linkage style input files:");
    string premesg = "", datmesg = "";
    for (Uint i = 0; i < prefile.size(); i++) {
      premesg += prefile[i] + " ";
      datmesg += datfile[i] + " ";
    }
    message("PREFILE " + premesg);
    message("DATFILE " + datmesg);
    message("");
    if (writeprobfiles) {
      Distribution::printprobfileswritten();
      message("");
    }
  }
  else if (decodeinput) {
    message("Using Decode style input files:");
    message("PREFILE " + prefile[0]);
    message("GENOTYPEFILE " + genotypefile.name);
    if (affectionfile.assigned())
      message("AFFECTIONFILE " + affectionfile.name);
    message("MAPFILE " + mapfile.name);
    message("FREQFILE " + freqfile.name);
    message("CHROMOSOMEFILE " + chromosomefile.name);
    message("");
    if (chromolist->all) 
      message("All chromosomes will be analyzed");
    else {
      message(string("The following chromosome") +
              (chromolist->next == 0 ? "" : "s") + " will be analysed:");
      string chr = "";
      for (Chromolist *cl = chromolist; cl != 0; cl = cl->next)
        chr = chr + cl->id + " ";
      message(chr);
    }
    message("");
  }
  else if (skipfirstphase && readprobfiles) {
    Distribution::printprobfilesread();
    message("");
  }
  // If simulations are to be performed show parameters
  if (simulation != 0) {
    message(string("SIMULATE dloc:") + simulation->diseasepos + " npre:" +
            simulation->nsimulations + " rep:" + simulation->nrepeats +
            " err:" + simulation->errorrate + " yield:" + simulation->yield +
            " het:" + simulation->hetrogen);
  }
  
  // Tell user which models are being run and the names of the output files
  int countanalyses = Model::modelcount();
  if (countanalyses == 1)
    message("The following analysis will be performed:");
  else if (countanalyses > 1)
    message("The following analyses will be performed:");
}
