#include "basic.h"

class ParseItem;
typedef vector<ParseItem *> ParseItems;

class ParseItem {
protected:
  string keyword;
  bool present;
public:
  virtual ~ParseItem() {}
  bool operator()() const {return present;}
  const string &getkeyword() const {return keyword;}

  ParseItem(const string &kw, ParseItems &items) :
      keyword(kw), present(false) {items.push_back(this);}
  ParseItem(const string &kw) : keyword(kw), present(false) {}  
  virtual bool parse(const string &word) = 0;
};

class ParseKeyword : public ParseItem {
public:
  ParseKeyword(const string &kw, ParseItems &items) : ParseItem(kw, items) {}
  ParseKeyword(const string &kw) : ParseItem(kw) {}
  virtual bool parse(const string &word) {
    if (word == keyword) {
      optassert(!present, "Keyword " + keyword +" used twice.");
      present = true;
      return true;
    }
    else return false;
  }
};

class ParseKeywordArgument : public ParseItem {
protected:
  string argument;
  
  bool keywordpresent(const string &word) {
    if (word.substr(0, keyword.length() + 1) == keyword + ":") return true;
    else return false;
  }
public:
  const string &getargument() const {return argument;}
  
  ParseKeywordArgument(const string &kw, ParseItems &items) :
      ParseItem(kw, items), argument("") {}
  ParseKeywordArgument(const string &kw) :
      ParseItem(kw), argument("") {}

  virtual bool parse(const string &word) {
    if (keywordpresent(word)) {
      optassert(!present, "Keyword " + keyword + " used twice.");
      present = true;
      argument = word.substr(keyword.length() + 1,
                             word.length() - keyword.length() - 1);
      return true;
    }
    else return false;
  }
};

class ParseNumeric : public ParseKeywordArgument {
protected:
  Double value;
  Double minvalue;
  Double maxvalue;

public:
  Double getvalue() const {return value;}
  
  ParseNumeric(const string &kw, ParseItems &items,
               Double defval, Double minval, Double maxval) :
      ParseKeywordArgument(kw, items), value(defval),
      minvalue(minval), maxvalue(maxval) {}
  ParseNumeric(const string &kw, Double defval, Double minval, Double maxval) :
      ParseKeywordArgument(kw), value(defval),
      minvalue(minval), maxvalue(maxval) {}
  virtual bool parse(const string &word) {
    if (ParseKeywordArgument::parse(word)) {
      value = atof(argument.c_str());
      optassert(value >= minvalue, string("Value ") + value + " too low for " +
                keyword + " (min is " + minvalue + ").");
      optassert(value <= maxvalue, string("Value ") + value + " too high for " +
                keyword + " (max is " + minvalue + ").");
      return true;
    }
    else return false;
  }
};

class ParseNumericVector : public ParseKeywordArgument {
protected:
  Doublevector values;
  Double minvalue;
  Double maxvalue;

  Uint minlength;
  Uint maxlength;
  
  bool keywordpresent(const string &word) {
    if (word.substr(0, keyword.length() + 1) == keyword + ":") return true;
    else return false;
  }
public:
  Double getvalue(Uint idx) const {
    optassert(present, "Numeric vector not set");
    optassert(idx < values.size(), "Too few arguments to " + keyword);
    return values[idx];
  }

  ParseNumericVector(const string &kw, ParseItems &items, Double minval,
                     Double maxval, Uint minlen, Uint maxlen) :
      ParseKeywordArgument(kw, items), minvalue(minval), maxvalue(maxval),
      minlength(minlen), maxlength(maxlen) {}
  ParseNumericVector(const string &kw, Double minval, Double maxval,
                     Uint minlen, Uint maxlen) :
      ParseKeywordArgument(kw), minvalue(minval), maxvalue(maxval),
      minlength(minlen), maxlength(maxlen) {}
  virtual bool parse(const string &word) {
    if (ParseKeywordArgument::parse(word)) {
      string w = argument;
      while (w.length() > 0) {
        string::size_type idx = w.find_first_of('/');
        Double value;
        if (idx == string::npos) {
          value = atof(w.c_str());
          w = "";
        }
        else {
          value = atof(w.substr(0, idx).c_str());
          w = w.substr(idx + 1, w.length() - idx - 1);
        }
        optassert(value >= minvalue, string("Value ") + value +
                  " too low for " + keyword + " (min is " + minvalue + ").");
        optassert(value <= maxvalue, string("Value ") + value +
                  " too high for " + keyword + " (max is " + minvalue + ").");
        values.push_back(value);
      }
      optassert(values.size() > 0, "No values given for " + keyword + ".");
      return true;
    }
    else return false;
  }
  Uint numvalues() const {return values.size();}
};

void parseline(ParseItems items, istream &ss, Stringvector &remainder) {
  while (!ss.eof()) {
    string word;
    ss >> word >> ws;
    bool found = false;
    for (Uint item = 0; item < items.size(); item++) {
      bool matches = items[item]->parse(word);
      optassert(!matches || !found,
                "Word " + word + " matches two keywords !!!");
      if (matches) found = true;
    }
    if (!found) remainder.push_back(word);
  }
}

Uint itemscount(const ParseItems &items) {
  Uint count = 0;
  for (Uint item = 0; item < items.size(); item++)
    if ((*items[item])()) count++;
  return count;
}

void checkexactlyone(const ParseItems &items, const string &linetype) {
  optassert(items.size() > 0, "Internal error !!");
  string possib = items[0]->getkeyword();
  for (Uint item = 1; item < items.size() - 1; item++)
    possib += ", " + items[item]->getkeyword();
  if (items.size() > 1)
    possib += " and " + items.back()->getkeyword();
  Uint count = itemscount(items);
  optassert(count > 0, "Exactly one of " + possib + " must be specified in " +
            linetype + " line.");
  optassert(count == 1, "At most one of " + possib + " must be specified in " +
            linetype + " line.");
}
