#include "files.h"
#include "peel.h"
#include "warning.h"
#include "vecutil.h"
#include "options.h"

void Peelbit::perform(IV v) {
  next->perform(v);
  next->perform(v | bit);
}

void Peelloop::perform(IV v) {
  for (int i = 0; i < proband->ngt; i++) {
    for (int j = 0; j < proband->ngt; j++)
      proband->peelprob[probandpeelnumber][j] = 0.0;
    proband->peelprob[probandpeelnumber][i] = 
      proband->peelprob[probandpeelnumber - 1][i];
    next->perform(v);
  }
}

void Peeldone::perform(IV v) {
  for (int i = 0; i < last->ngt; i++) L[v] += last->finalprob[i];
}

void Peel2child::perform(IV v) {
  for (int p = 0; p < proband->ngt; p++)
    proband->peelprob[probandpeelnumber][p] = 0.0;
  // loop through fathers gentotypes
  for (int f = 0; f < marriage->husband->ngt; f++) {
    // loop through mother genotypes
    for (int m = 0; m < 4; m++) {
      Double prob = marriage->husband->finalprob[f]*
        marriage->wife->finalprob[m];
      int probandgt = -1;
      // loop through children
      for (ESchildren *c = marriage->child; c != 0; c = c->next) {
        int kidgt;
        if (!options->sexlinked || c->val->per->sex == FEMALE)
          kidgt = c->val->allelesfromparents(v, f, m);
        else kidgt = c->val->allelefrommother(v, m);
        if (c->val == proband) {
          probandgt = kidgt;
          prob *= proband->peelprob[probandpeelnumber - 1][kidgt];
        } else prob *= c->val->finalprob[kidgt];
      }
      proband->peelprob[probandpeelnumber][probandgt] += prob;
    }
  }
  next->perform(v);
}

void Peel2parent::perform(IV v) {
  int kidgt = 0;
  ESperson *spouse = (marriage->husband == proband) ? 
    marriage->wife : marriage->husband;
  // loop through proband genotypes
  for (int p = 0; p < proband->ngt; p++) {
    Double tmpprob = 0.0;
    // loop through spouse genotypes
    for (int s = 0; s < spouse->ngt; s++) {
      Double prob = spouse->finalprob[s];
      for (ESchildren *c = marriage->child; c != 0; c = c->next) {
        if (!options->sexlinked || c->val->per->sex == FEMALE)
          kidgt = (marriage->husband == proband) ?
            c->val->allelesfromparents(v, p, s)
            : c->val->allelesfromparents(v, s, p);
        else
          kidgt = (marriage->husband == proband) ?
            c->val->allelefrommother(v, s)
            : c->val->allelefrommother(v, p);
        prob *= c->val->finalprob[kidgt];
      }
      tmpprob += prob;
    }
    proband->peelprob[probandpeelnumber][p] = 
      proband->peelprob[probandpeelnumber - 1][p]*tmpprob;
  }
  next->perform(v);
}

ESgraph::ESgraph(Family &fam, Trait *t) {
  numiv = fam.numiv;
  trait = t;
  bitmask = 0;
  // Create people nodes
  firstmar = 0;
  peelorder = lastpeel = 0;
  first = new ESpeople(new ESperson(fam.first), &first, 0);
  for (Person *p = fam.first->next; p != 0; p = p->next) {
    first->insertafter(new ESperson(p));
  }
  // Create marriages and edges
  for (ESpeople *n = first; n != 0; n = n->next)
    n->val->addparentalmarriage(*this);
  trimuninformative();
//    assertcond(first != 0, string("Families with no informative people ") +
//               "should be thrown out earlier");
  if (firstmar == 0 && first != 0 && first->next != 0)
    warning(string("Familes with disjo") +
            "int sets of informative people should be thrown out earlier");
}

void ESgraph::trimuninformative() {
  for (ESmarriages *m = firstmar; m != 0;) {
    ESmarriage *mar = m->val;
    for (ESchildren *c = mar->child; c != 0;) {
      if (c->val->uninformativedstat() && c->val->mar == 0) {
        bitmask |= c->val->per->matmask | c->val->per->patmask;
        m->val->bitmask &= ~bitmask;
        removeperson(c->val);
        delete c;
        c = mar->child;
      } else c = c->next;
    }
    if (mar->husband->parmar == 0 && mar->wife->parmar == 0 &&
        mar->husband->mar->next == 0 && mar->wife->mar->next == 0 &&
        mar->husband->uninformativedstat() && mar->wife->uninformativedstat() &&
        (mar->child == 0 || mar->child->next == 0)) {
      removeperson(mar->husband);
      removeperson(mar->wife);
      mar->husband = mar->wife = 0;
    }
    if (mar->husband == 0 && mar->wife == 0 || mar->child == 0) {
      if (mar->husband != 0) {
        mar->husband->removemarriage(mar);
        if (mar->husband->uninformativedstat() &&
            mar->husband->parmar == 0 && mar->husband->mar == 0) {
          removeperson(mar->husband);
        }
      }    
      if (mar->wife != 0) {
        mar->wife->removemarriage(mar);
        if (mar->wife->uninformativedstat() &&
            mar->wife->parmar == 0 && mar->wife->mar == 0) {
          removeperson(mar->wife);
        }
      }    
      for (ESchildren *c = mar->child; c != 0; c = c->next) {
        c->val->parmar = 0;
        bitmask |= c->val->per->matmask | c->val->per->patmask;
      }
      delete m;
      m = firstmar;
    } else m = m->next;
  }
}

ESmarriage::ESmarriage(ESperson *h, ESperson *w, ESperson *c) {
  peeled = ripe = false;
  bitmask = bitcount = 0;
  husband = h; 
  wife = w; 
  child = 0;
  if (husband->mar == 0) 
    husband->mar = new ESmarriages(this, &(husband->mar), husband->mar);
  else husband->mar->insertafter(this);
  if (wife->mar == 0)
    wife->mar = new ESmarriages(this, &(wife->mar), wife->mar);
  else wife->mar->insertafter(this);
  addchild(c);
}

ESmarriage *ESperson::marriedto(ESperson *spouse) const {
  for (ESmarriages *m = mar; m != 0; m = m->next) 
    if (m->val->wife == spouse || m->val->husband == spouse)
      return m->val;
  return 0;
}

ESperson::ESperson(Person *p) :
    parmar(0), mar(0), per(p), split(false), flag(false), peelcount(1),
    finalprob(0), peelprob(0) {
  ngt = !options->sexlinked || per->sex == FEMALE ? 4 : 2;
}

int ESperson::allelesfromparents(IV v, int f, int m) const {
  bool pp = options->sexlinked ? true : v & per->patmask;
  return (v & per->matmask ? m%2 : m/2) + 2*(pp ? f%2 : f/2);
}

void ESperson::removemarriage(ESmarriage *rm)  {
  for (ESmarriages *m = mar; m != 0;) {
    ESmarriages *next = m->next;
    if (m->val == rm) delete m;
    m = next;
  }
}

ESperson *ESgraph::findperson(Person *p) const {
  for (ESpeople *q = first; q != 0; q = q->next) 
    if (q->val->per == p) return q->val;
  return 0;
}

void ESgraph::removeperson(ESperson *p) {
  for (ESpeople *q = first; q != 0;) {
    ESpeople *next = q->next;
    if (q->val == p) {
      delete q;
      delete p;
    }
    q = next;
  }
}

void ESperson::addparentalmarriage(ESgraph &g) {
  if (per->father != 0) { // then also mother != 0 
    // see if marriage between father and mother allready exists
    ESperson *dad = g.findperson(per->father);
    ESperson *mom = g.findperson(per->mother);
    parmar = dad->marriedto(mom);
    if (parmar == 0) {
      parmar = new ESmarriage(dad, mom, this);
      g.addmarriage(parmar);
    } else 
      parmar->addchild(this);
  }
}

bool ESmarriage::hasloops(ESperson *caller) {
  if (husband != 0 && husband != caller && husband->hasloops(this)) return true;
  if (wife != 0 && wife != caller && wife->hasloops(this)) return true;
  for (ESchildren *c = child; c != 0; c = c->next) 
    if (c->val != caller && c->val->hasloops(this)) return true;
  return false;
}

bool ESperson::hasloops(ESmarriage *caller) {
  if (flag) return true;
  flag = true;
  if (parmar != 0 && parmar != caller && parmar->hasloops(this)) return true;
  for (ESmarriages *m = mar; m != 0; m = m->next)
    if (m->val != caller && m->val->hasloops(this)) return true;
  return false;
}

void ESgraph::peel(DoubleVec S) {
  if (first == 0 || firstmar == 0)
    copyval(S, 1.0, numiv);
//    assertcond(firstmar != 0,
//               "Families with only one affected should be dropped earlier");
  else {
    createpeelorder();
    // Allocate memory for peelprob property of ESperson and set 
    // the inital probabilites to their appropriate values
    initializeprob();
    Peelaction *pa;
    for (pa = peelorder; pa->next != 0; pa = pa->next);
    ((Peeldone *)pa)->L = S;
    peelorder->perform(0);
    expandiv(S, bitmask, numiv);
  }
}

void ESgraph::createpeelorder() {
  ESmarriages *ellegible = 0;
  for (ESmarriages *m = firstmar; m != 0; m = m->next) {
    if (m->val->isripe()) {
      m->val->ripe = true;
      if (ellegible == 0)
        ellegible = new ESmarriages(m->val, &ellegible, ellegible);
      else ellegible->insertafter(m->val);
    }
  }
  ESperson *proband = 0;
  bool done = false;
  while (!done) {
    // See if all branches have been peeled, and only a loop remains
    if (ellegible == 0) {
      // No marriage to peel, so we must break a loop

      // First find some person in a loop, i.e. anyone who is not a
      // leaf
      ESpeople *p = first;
      for (; p != 0 && p->val->isleaf(); p = p->next);
      assertinternal(p != 0);
      Peelloop *nextpeel = new Peelloop;
      nextpeel->proband = p->val;
      nextpeel->proband->split = true;
      nextpeel->proband->peelcount++;
      nextpeel->probandpeelnumber = nextpeel->proband->peelcount - 1;
      addpeel(nextpeel);
    } else {
      // There is a marriage to be peeled

      // First find cheapest marriage to peel
      int maxbits = -1;
      ESmarriages *minell = 0;
      for (ESmarriages *e = ellegible; e != 0; e = e->next) {
        if (maxbits < e->val->bitcount) {
          maxbits = e->val->bitcount;
          minell = e;
        }
      }
      assertinternal(minell != 0);
      // Add maxbits number of Peelbits to peelorder
      for (int i = 0; i < 32; i++) {
        if (POW2[i] & minell->val->bitmask) {
          Peelbit *nextpeel = new Peelbit();
          addpeel(nextpeel);
          nextpeel->bit = POW2[i];
        }
      }
      // Next identify proband and add the marriage to the peelorder
      proband = 0;
      if (!minell->val->wife->split) 
        proband = minell->val->wife;
      if ((proband == 0 || !minell->val->husband->isleaf()) && 
          !minell->val->husband->split) 
        proband = minell->val->husband;
      for (ESchildren *c = minell->val->child; c != 0; c = c->next)
        if ((proband == 0 || !c->val->isleaf()) && !c->val->split) 
          proband = c->val;
      assertcond(proband != 0, 
                 "Can not find a proband to peel. Try removing some loops.");
      proband->peelcount++;
      Peelmarriage *nextpeel;
      if (proband == minell->val->husband || 
          proband == minell->val->wife) nextpeel = new Peel2parent;
      else nextpeel = new Peel2child;
      nextpeel->probandpeelnumber = proband->peelcount - 1;
      addpeel(nextpeel);
      nextpeel->proband = proband;
      nextpeel->marriage = minell->val;
      // Finally remove peeled marriage from ellegible list 
      minell->val->peeled = true;
      delete minell;
    }
    // Find all ellegible marriages
    done = true;
    for (ESmarriages *m = firstmar; m != 0; m = m->next) {
      done = done && m->val->peeled;
      if (m->val->isripe()) {
        m->val->ripe = true;
        if (ellegible == 0)
          ellegible = new ESmarriages(m->val, &ellegible, ellegible);
        else ellegible->insertafter(m->val);
      }
    }
  }
  // Put in a Pelldone action at the end of the peeling order to
  // wrap things up
  if (proband == 0)
    assertinternal(proband != 0);
  Peeldone *lastpeel = new Peeldone;
  lastpeel->last = proband;
  addpeel(lastpeel);
}

void ESgraph::initializeprob() {
  // initprobs is a three dimensional array
  // the first index is 1 for founders, 0 for decendants
  // the second index is for affection status
  // the third index runs through the 3 possible genotypes
  Double *initprobs[2][3][3];
  for (int a = 0; a < 2; a++)
    for (int b = 0; b < 3; b++)
      for (int c = 0; c < 3; c++)
        initprobs[a][b][c] = new Double[trait->nliability];

  DoubleMat normconst0;
  DoubleMat normconst1;
  NEWMAT(Double, normconst0, 3, trait->nliability);
  NEWMAT(Double, normconst1, 3, trait->nliability);
  zero(normconst0[0], 3*trait->nliability);
  zero(normconst1[0], 3*trait->nliability);
  
  for (Uint l = 0; l < trait->nliability; l++) {
    if (options->sexlinked && (l % 2) == 1) {
      for (int i = 0; i < 2; i++) {
        Double pen = 0.0;
        Double prior = 0.0;
        switch (i) {
          case 0: 
            pen = trait->penetrance[l][0];
            prior = trait->disfreq;
            break;
          case 1:
            pen = trait->penetrance[l][1]; 
            prior = 1.0 - trait->disfreq;
            break;
        }
        initprobs[0][0][i][l] = .25;
        initprobs[0][1][i][l] = 1 - pen;
        initprobs[0][2][i][l] = pen;
        initprobs[1][0][i][l] = prior;
        initprobs[1][1][i][l] = prior*(1 - pen);
        initprobs[1][2][i][l] = prior*pen;
        for (int j = 0; j < 3; j++) {
          normconst0[j][l] += initprobs[0][j][i][l];
          normconst1[j][l] += initprobs[1][j][i][l];
        }
      }
      for (int i = 0; i < 2; i++)
        for (int j = 0; j < 3; j++) {
          initprobs[0][j][i][l] /= normconst0[j][l];
          initprobs[1][j][i][l] /= normconst1[j][l];
        }
    }
    else {
      for (int i = 0; i < 3; i++) {
        Double pen = 0.0;
        Double prior = 0.0;
        switch (i) {
          case 0: 
            pen = trait->penetrance[l][0];
            prior = trait->disfreq*trait->disfreq;
            break;
          case 1:
            pen = trait->penetrance[l][1]; 
            prior = trait->disfreq*(1.0 - trait->disfreq);
            break;
          case 2: 
            pen = trait->penetrance[l][2]; 
            prior = (1.0 - trait->disfreq)*(1.0 - trait->disfreq);
            break;
        }
        initprobs[0][0][i][l] = .25;
        initprobs[0][1][i][l] = 1 - pen;
        initprobs[0][2][i][l] = pen;
        initprobs[1][0][i][l] = prior;
        initprobs[1][1][i][l] = prior*(1 - pen);
        initprobs[1][2][i][l] = prior*pen;
        Double multiplier = i == 1 ? 2.0 : 1.0;
        for (int j = 0; j < 3; j++) {
          normconst0[j][l] += multiplier*initprobs[0][j][i][l];
          normconst1[j][l] += multiplier*initprobs[1][j][i][l];
        }
      }
      for (int i = 0; i < 3; i++)
        for (int j = 0; j < 3; j++) {
          initprobs[0][j][i][l] /= normconst0[j][l];
          initprobs[1][j][i][l] /= normconst1[j][l];
        }
    }
  }
  DELETEMAT(normconst0);
  DELETEMAT(normconst1);
  for (ESpeople *p = first; p != 0; p = p->next) {
    int pc = p->val->peelcount;
    NEWMAT(Double, p->val->peelprob, pc, 4);
    p->val->finalprob = p->val->peelprob[pc - 1];
    int founder = p->val->parmar == 0 ? 1 : 0;
    Diseasestatus dis = p->val->per->dstat;
    Uint liab = p->val->per->liability;
    p->val->peelprob[0][0] = initprobs[founder][dis][0][liab];
    p->val->peelprob[0][1] = p->val->peelprob[0][2] = 
      initprobs[founder][dis][1][liab];
    p->val->peelprob[0][3] = initprobs[founder][dis][2][liab];
  }
}  

bool ESmarriage::isripe() const {
  if (ripe || peeled) return false;
  int nonleaves = 0;
  if (!husband->isleaf()) nonleaves++;
  if (!wife->isleaf()) nonleaves++;
  for (ESchildren *c = child; 
       c != 0 && nonleaves < 2; c = c->next)
    if (!c->val->isleaf()) nonleaves++;
  return nonleaves < 2;
}

void ESmarriage::addchild(ESperson *c) {
  if (child == 0) 
    child = new ESchildren(c, &child, child);  
  else child->insertafter(c);
  bitcount += (c->per->patmask ? 1 : 0) + (c->per->matmask ? 1 : 0);
  bitmask |= c->per->patmask | c->per->matmask;
}

void ESgraph::addpeel(Peelaction *p) {
  if (lastpeel == 0) peelorder = lastpeel = p;
  else {
    lastpeel->next = p; 
    lastpeel = p;
  }
}

bool ESperson::isleaf() const {
  if (split) return true;
  int notpeeled = 0;
  notpeeled += parmar != 0 && !parmar->peeled ? 1 : 0;
  for (ESmarriages *m = mar; m != 0 && notpeeled < 2; m = m->next)
    notpeeled += !m->val->peeled ? 1 : 0;
  return notpeeled < 2;
}
