#include "simulation.h"
#include "files.h"
#include "peel.h"
#include "singlelocus.h"
#include "vecutil.h"
#include "haldane.h"
#include "map.h"
#include "family.h"
#include "trait.h"
#include "options.h"
#include "rgamma.h"

Double runif() {
  return erand48(options->seed);
}

void Simulation::setfoundergenotypes(const Map &map) {
  for (Uint gam = 0; gam < map.num; gam++) {
    Uint a = 1;
    for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
      if (perfectdata) {
        p->gen[0][gam] = a++;
        if (!options->sexlinked || p->sex == FEMALE) p->gen[1][gam] = a++;
        else p->gen[1][gam] = p->gen[0][gam];
      }
      else {
        p->gen[0][gam] = map.randomallele(gam);
        if (!options->sexlinked || p->sex == FEMALE)
          p->gen[1][gam] = map.randomallele(gam);
        else p->gen[1][gam] = p->gen[0][gam];
      }
    }
  }
}

void Simulation::simulate(Family *famp, const Map &map) {
//  if (perfectdata)
    for (Person *p = famp->first; p != 0; p = p->next)
      if (!p->genotyped)
        p->allocgenotypes(map.num);

  fam = famp;
  if (diseasepos >= 0) {
    zero(L, fam->numiv);
    ESgraph g = ESgraph(*fam, trait);
    g.peel(L);
    Double cumsum = 0.0; 
    for (IV w = 0; w < fam->numiv; w++) {
      cumsum += L[w];
      L[w] = cumsum;
    }
  }
  ios::openmode mode = firstfam ? ios::out | ios::trunc : ios::out | ios::app;
  firstfam = false;
  for (int s = 0; s < nsimulations; s++) {
    string postfix = "";
    for (Uint i = Uint(log10(Double(s + 1)));
         i < Uint(log10(Double(nsimulations))); i++)
      postfix += "0";
    string filename = options->prefile[0] + "." + postfix + (s + 1);
    ofstream out(filename.c_str(), mode);
    for (int i = 0; i < nrepeats; i++) {
      // pick an IV at the disease locus
      IV v;
      if (diseasepos >= 0 && runif() >= hetrogen) {
        Double r = runif()*L[fam->numiv - 1];
        for (v = 0; v < fam->numiv && L[v] <= r; v++);
      }
      else v = IV(nrand48(options->seed)) % fam->numiv;
      // simulate founder alleles
      setfoundergenotypes(map);
      // simulate from disease locus
      simulatefromdiseaselocus(map, v);
      // print out results
      for (Person *p = fam->first; p != 0; p = p->next) {
        out << fam->id;
        if (nrepeats > 1) out << "-" << i + 1;
        out <<  "  " << p->id << " "
            << (p->father == 0 ? "0" : p->father->id) << " "
            << (p->mother == 0 ? "0" : p->mother->id) << " "
            << p->sex + 1 << " " << p->dstat;
        for (Uint gam = 0; gam < map.num; gam++)
          out << " " << p->gen[0][gam] << " " << p->gen[1][gam];
        out << "\n";
      }
      out.flush();
    }
  }
}

void Simulation::simulatefromdiseaselocus(const Map &map, IV v) {
  // find the marker on the left of the disease locus
  Uint leftmarker = 0;
  if (diseasepos >= 0) {
    for (; leftmarker < map.num; leftmarker++)
      if (map.markerpos[0][leftmarker] >= diseasepos) break;
    if (map.markerpos[0][leftmarker] > diseasepos) {
      assertinternal(leftmarker > 0);
      leftmarker--;
    }
  }
  // Calculate sex-specific disease positions
  Float diseasepos_male;
  Float diseasepos_female;
  if (diseasepos < 0)
    diseasepos_female = diseasepos_male = 0;
  else if (diseasepos > map.markerpos[0][map.num - 1]) {
    diseasepos_male = map.markerpos[1][map.num - 1];
    diseasepos_female = map.markerpos[2][map.num - 1];
  }
  else {
    Float d = ((diseasepos - map.markerpos[0][leftmarker])/
               (map.markerpos[0][leftmarker + 1] -
                map.markerpos[0][leftmarker]));
    diseasepos_male = map.markerpos[1][leftmarker] +
      d*(map.markerpos[1][leftmarker + 1] - map.markerpos[1][leftmarker]);
    diseasepos_female = map.markerpos[2][leftmarker] +
      d*(map.markerpos[2][leftmarker + 1] - map.markerpos[2][leftmarker]);
  }
  if (diseasepos >= 0) {
    // Initialize crossover process
    FloatVec left_paticd;
    FloatVec left_maticd;
    FloatVec right_paticd;
    FloatVec right_maticd;
    NEWVEC(Float, left_paticd, fam->numnf);
    NEWVEC(Float, left_maticd, fam->numnf);
    NEWVEC(Float, right_paticd, fam->numnf);
    NEWVEC(Float, right_maticd, fam->numnf);
    Uint i = 0;
    for (Person *p = fam->firstdescendant; p != 0; p = p->next, i++) {
      initicd(diseasepos_male, nu_male, left_paticd[i], right_paticd[i]);
      initicd(diseasepos_female, nu_female, left_maticd[i], right_maticd[i]);
    }
    // Simulate left
    inittrans(v);
    simulateicd(map, leftmarker, left_paticd, left_maticd, LEFT);
    // Simulate right
    inittrans(v);
    simulateicd(map, leftmarker + 1, right_paticd, right_maticd, RIGHT);
    DELETEVEC(left_paticd);
    DELETEVEC(left_maticd);
    DELETEVEC(right_paticd);
    DELETEVEC(right_maticd);
  }
  else {
    // Initialize crossover process
    FloatVec paticd;
    FloatVec maticd;
    NEWVEC(Float, paticd, fam->numnf);
    NEWVEC(Float, maticd, fam->numnf);
    Uint i = 0;
    for (Person *p = fam->firstdescendant; p != 0; p = p->next, i++) {
      paticd[i] = 100.*rgammamix_end(nu_male);
      maticd[i] = 100.*rgammamix_end(nu_female);
    }
    inittrans(v);
    simulateicd(map, 0, paticd, maticd, RIGHT);
    DELETEVEC(paticd);
    DELETEVEC(maticd);
  }
  // Clean up
}

void Simulation::simulatemarker(const Map &map, int gam) {
  // set founder node alleles to the value at marker gam
  for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
    p->nod[0]->allele = p->gen[0][gam];
    p->nod[1]->allele = p->gen[1][gam];
  }
  // set people's nodes
  Uint i = 0;
  for (Person *p = fam->firstdescendant; p != 0; p = p->next, i++) {
    p->nod[1] = p->mother->nod[mattrans[i]];
    if (options->sexlinked && p->sex == MALE) 
      p->nod[0] = p->nod[1];
    else
      p->nod[0] = p->father->nod[pattrans[i]];
  }
  // set people's alleles
  for (Person *p = fam->first; p != 0; p = p->next) {
    if (perfectdata || (p->genotyped && runif() <= yield)) {
      if (runif() > errorrate) {
        p->gen[0][gam] = p->nod[0]->allele;
        p->gen[1][gam] = p->nod[1]->allele;
      }
      else {
        p->gen[0][gam] = map.randomallele(gam);
        p->gen[1][gam] = map.randomallele(gam);
      }
    }
    else p->gen[0][gam] = p->gen[1][gam] = ALLELEUNKNOWN;
  }
}

void Simulation::initicd(Double diseasepos, Double nu,
                         Float &left_icd, Float &right_icd) {
//   Double r = runif();
//   Double d = 100*rgammamix(nu);
//   left_icd = diseasepos - r*d;
//   right_icd = diseasepos + (1 - r)*d;
  const Double l = 100.*rgammamix_end(nu);
  Double r;
  do {
    r = 100.*rgammamix(nu) - l;
  } while (r < 0);
  left_icd = diseasepos - l;
  right_icd = diseasepos + r;
}

void Simulation::inittrans(IV v) {
  // set non founder nodes to their value according to v
  Uint i = 0;
  for (Person *p = fam->firstdescendant; p != 0; p = p->next, i++) {
    pattrans[i] = (p->patmask & v) > 0 ? 1 : 0;
    mattrans[i] = (p->matmask & v) > 0 ? 1 : 0;
  }
}

void Simulation::simulateicd(const Map &map, Uint from,
                             FloatVec paticd, FloatVec maticd, int dir) {
  for (int gam = from; gam >= 0 && gam < map.num; gam += dir) {
    // Simulate crossover process
    int i = 0;
    for (Person *p = fam->firstdescendant; p != 0; p = p->next, i++) {
      while (dir*paticd[i] < dir*map.markerpos[1][gam]) {
        paticd[i] += dir*100*rgammamix(nu_male);
        pattrans[i] ^= 1;
      }
      while (dir*maticd[i] < dir*map.markerpos[2][gam]) {
        maticd[i] += dir*100*rgammamix(nu_female);
        mattrans[i] ^= 1;
      }
    }
    // Finally simulate people's genotypes
    simulatemarker(map, gam);
  }
}
