#include <iostream>
#include <fstream>
#include <iomanip>
#include "singlelocus.h"
#include "probability.h"
#include "files.h"
#include "options.h"
#include "vecutil.h"
#include "matrix.h"
#include "map.h"
#include "family.h"

class Elist : public Plist {
public:
  Elist(Person *q, Plist *ch) : Plist(q, ch) {};
  ~Elist() {delete next;}
  
  Edge *edge;
};

bool hasgenotypeddesc(Person *p, Uint gam) {
  bool res = false;
  for (Plist *c = p->children; c != 0 && !res; c = c->next) {
    res |= c->p->genotyped && c->p->genotyp[gam];
    if (!res) res |= hasgenotypeddesc(c->p, gam);
  }
  return res;
}

bool Probability::checkq(Uint gam) {
  bool allzeros = true;
  bool informat = false;
  FloatVec curq = q->getrow(gam, false);
  Float c = curq[0];
  for (IV v = 0; v < numiv && (allzeros || !informat); v++) {
    if (allzeros) allzeros = curq[v] == 0.0;
    if (!informat) informat = curq[v] != c;
  }
  if (!informat) markuninformative(gam);
  else informative[gam] = true;
  if (allzeros)
    if (options->inconsistent.assigned()) {
      options->inconsistent << map->markername[gam] << "\t" << fam->id << "\n";
      informative[gam] = false;
      copyval(curq, 1.0/Double(numiv), numiv);
    }
    else
      assertcond(false, "Family " + fam->id + " has inconsistent genotypes at " +
                 "marker " + map->markername[gam]);
  return informat;
}

Graph::Graph(Person *frst) : allelefreq(0), first(0) {
  for (Person *p = frst; p != 0 && p->founder(); p = p->next) {
    p->nod[0] = addnode();
    if (!options->sexlinked || p->sex == FEMALE) p->nod[1] = addnode();
    else p->nod[1] = p->nod[0];
  }
}
void Probability::createfoundergraph() {
  graph = new Graph(fam->first);
  if (options->qfile.assigned()) options->qfile.open();
}

void Probability::markuninformative(Uint gam) {
  assertinternal(gam < map->num);
  if (informative[gam]) {
    if (options->uninformative.assigned())
      options->uninformative << map->markername[gam] << "\t" << fam->id << "\n";
    informative[gam] = false;
  }
}

Float Probability::fromhere(Uint gam, FloatVec qp, FloatVec freq, Family *fam) {
  assertinternal(graph != 0);
  zero(qp, fam->numiv);
  graph->reset();
  graph->setallelefreq(freq);
  Float nc = 1.0;
  // Add genotype edges belonging to founders
  for (Person *p = fam->first; p != fam->firstdescendant; p = p->next) {
    if (p->gen[0] != 0 && p->genotyp[gam]) {
      Allele a0 = p->gen[0][gam];
      Allele a1 = p->gen[1][gam];
      if (a0 == a1) {
        if (p->nod[0] != p->nod[1]) nc *= freq[a0 - 1]*freq[a0 - 1];
        else nc *= freq[a0 - 1];
      }
      graph->addfounderedge(p->nod[0], p->nod[1], a0, a1);
    }
  }
  // Throw out ungenotyped leaves
  Plist *firstper = 0, *lastper = 0;
  Uint uninformativemask = 0;
  for (Person *p = fam->firstdescendant; p != 0; p = p->next)
    if (p->children != 0 && hasgenotypeddesc(p, gam) ||
        p->genotyped && p->genotyp[gam]) {
      if (firstper == 0) lastper = firstper = new Elist(p, 0);
      else {
        lastper->next = new Elist(p, 0);
        lastper = lastper->next;
      }
      if (p->gen[0] != 0 && p->genotyp[gam])
        ((Elist *)(lastper))->edge =
          new Edge(p->gen[0][gam], p->gen[1][gam], freq);
    } else uninformativemask |= p->patmask | p->matmask;
  // Calculate single locus probabilities
  findq(firstper, qp, gam, 0, graph);
  // Fill in singlelocus probs for all IV's
  expandiv(qp, uninformativemask, fam->numiv);
  nc *= normal<Float>(qp, fam->numiv);
  // Output results
  if (options->qfile != 0 && gam <= options->maxlocusforoutput) 
    outputvec(options->qfile, gam, qp, fam->numiv);
  delete firstper;
  graph->reset();
  return nc;
}

void Probability::deletefoundergraph() {
  delete graph;
}

void Probability::flattenq(int gam, Float r) {
  // Flattens the singlelocus distribution at gam
  FloatVec curq = q->getrow(gam, false);
  if (r == 0.0) copyval(curq, 1.0/Float(numiv), numiv);
  else if (r < 1.0) {
    Float c = sum<Float>(curq, numiv);
    Float flat = (1.0 - r)/r*c/Float(numiv);
    for (IV v = 0; v < numiv; v++) curq[v] += flat;
  } // else r is 1 and nothing needs to be done
}

void Probability::findq(Plist* pl, FloatVec q, int gam, IV v, Graph *graph) {
  // v = inheritance vector
  if (pl == 0) q[v] += graph->probability();
  else {
    Person *p = pl->p;
    int Kf = p->patmask ? 1 : 0;
    int Km = p->matmask ? 1 : 0;
    bool n0set, n1set;

    for (int K1 = 0; K1 <= Km; K1++) {
      if (K1) v |= p->matmask;
      Node *n1 = p->mother->nod[K1];
      p->nod[1] = n1;
      for (int K0 = 0; K0 <= Kf; K0++) {
        if (K0) v |= p->patmask;
        Node *n0 = p->father->nod[K0];
        if (!options->sexlinked || p->sex == FEMALE) p->nod[0] = n0;
        else n0 = p->nod[0] = p->nod[1];
        if (p->gen[0] != 0 && p->genotyp[gam]) {
          Allele a0 = p->gen[0][gam];
          Allele a1 = p->gen[1][gam];
          if (a0 == a1) { // homozygous
            if (graph->addgenotype(n0, n1, a0, a1, n0set, n1set)) {
              findq(pl->next, q, gam, v, graph);
              graph->removegenotype(n0, n1, n0set, n1set);
            }
          }
          else if (n0 != n1) { // heterozygous (alleles must have come from
                               // different sources)
            if (n0->allele == ALLELEUNKNOWN && n1->allele == ALLELEUNKNOWN &&
                n0->edge == 0 && n1->edge == 0) {
              graph->addedge(n0, n1, ((Elist *)(pl))->edge);
              findq(pl->next, q, gam, v, graph);
              graph->removeedge(n0, n1);
            }
            else {
              if (graph->addgenotype(n0, n1, a0, a1, n0set, n1set)) {
                findq(pl->next, q, gam, v, graph);
                graph->removegenotype(n0, n1, n0set, n1set);
              }
              if (graph->addgenotype(n0, n1, a1, a0, n0set, n1set)) {
                findq(pl->next, q, gam, v, graph);
                graph->removegenotype(n0, n1, n0set, n1set);
              }
            }
          }
        }
        else findq(pl->next, q, gam, v, graph);
      }
      v &= ~p->patmask;
    }
  }
}

////////////////////////////////////////////////////////////
// Graph classes

Graph::~Graph() {
//  deleteedges();
  Node *tmp;
  for (Node *cur = first; cur != NULL;) {
    tmp = cur->next;
    delete cur;
    cur = tmp;
  }
}

void Graph::reset() {
  for (Node *cur = first; cur != NULL; cur = cur->next) {
    cur->allele = ALLELEUNKNOWN;
    if (cur->neighbour) {
      delete cur->edge;
      cur->edge = cur->neighbour->edge = 0; 
      cur->neighbour = 0;
    }
    cur->genotyped = false;
  }
}

void Graph::haploreset() {
  for (Node *cur = first; cur != NULL; cur = cur->next) {
    cur->allele = ALLELEUNKNOWN;
    for (Edge *e = cur->edge; e != 0;) {
      Edge *tmp = e->next;
      delete e;
      e = tmp;
    }
    cur->edge = 0;
  }
}

Node *Graph::addnode() {
  first = new Node(first);
  if (first->next == 0) first->index = 0;
  else first->index = first->next->index + 1;
  return first;
}

void Graph::addfounderedge(Node *n0, Node *n1, Allele b0, Allele b1) const {
  if (b0 == b1 || n0 == n1) {
    assertcond(b0 == b1, string("Trying to assign a heterozygous genotype to") +
               " one founderallele");
    n0->allele = n1->allele = b0;
    n1->genotyped = n0->genotyped = true;
  }
  else addedge(n0, n1, new Edge(b0, b1, allelefreq));
}

bool Graph::addgenotype(Node *n0, Node *n1, Allele a0, Allele a1) {
  if (n0->allele == ALLELEUNKNOWN && n1->allele == ALLELEUNKNOWN && a0 != a1) {
    // Then Try to add an edge
    bool n0canaddedge = (n0->edge == 0 ||
                         (n0->edge->a0 == a0 && n0->edge->a1 == a1) ||
                         (n0->edge->a0 == a1 && n0->edge->a1 == a0));
    bool n1canaddedge = (n1->edge == 0 || 
                         (n1->edge->a0 == a0 && n1->edge->a1 == a1) ||
                         (n1->edge->a0 == a1 && n1->edge->a1 == a0));
    if (n0canaddedge && n1canaddedge) {
      Edge *e0 = new Edge(a0, a1, n0, n1);
      Edge *e1 = new Edge(a0, a1, n1, n0);
      e0->next = n0->edge;
      n0->edge = e0;
      e1->next = n1->edge;
      n1->edge = e1;
      return true;
    }
    else {
      if (n0->haplosetallele(a0) && n1->haplosetallele(a1)) return true;
      else {
        n0->clearallele();
        n1->clearallele();
        return n0->haplosetallele(a1) && n1->haplosetallele(a0);
      }
    }
  }
  else {
    // Try to assign the alleles
    if (n0->haplosetallele(a0) && n1->haplosetallele(a1)) return true;
    else {
      n0->clearallele();
      n1->clearallele();
      return n0->haplosetallele(a1) && n1->haplosetallele(a0);
    }
  }
}

void Graph::assignalleles(bool pickmle) {
  extern Double runif();
  for (Node *n = first; n != 0; n = n->next)
    if (n->allele == ALLELEUNKNOWN && n->edge != 0) {
      n->haplosetallele(n->edge->a0);
      Float p0 = probability();
      n->clearallele();
      n->haplosetallele(n->edge->a1);
      Float p1 = probability();
      if (pickmle && p0 > p1 ||
          !pickmle && runif()*(p0 + p1) < p0) {
        n->clearallele();
        n->haplosetallele(n->edge->a0);
      }
    }
}

void Node::getBlock(vector<unsigned int> &b1, vector<unsigned int> &b2) {
  if (allele == ALLELEUNKNOWN) {
    assertinternal(edge != 0);
    allele = Allele(-1);
    b1.push_back(index);

    for (Edge *e = edge; e != 0; e = e->next)
      e->n1->getBlock(b2, b1);
  }
}

void Node::clearallele() {
  if (allele != ALLELEUNKNOWN) {
    allele = ALLELEUNKNOWN;
    for (Edge *e = edge; e != 0; e = e->next)
      e->n1->clearallele();
  }
}

bool Node::haplosetallele(Allele a) {
  if (allele != ALLELEUNKNOWN) return a == allele;
  if (edge == 0) {
    allele = a;
    return true;
  }
  if (edge->a0 == a || edge->a1 == a) {
    allele = a;
    Allele b = edge->a0 == a ? edge->a1 : edge->a0;
    for (Edge *e = edge; e != 0; e = e->next)
      assertcond(e->n1->haplosetallele(b),
                 "Haplotype allele assignment impossible");
    return true;
  }
  else return false;
}
