#ifndef _VECTORUTILS
#define _VECTORUTILS
#include "basic.h"

void fft(FloatVec d, FloatVec s, Uint N, Uint nbits); // fast Fourier transform

template<class T, class N, class Tp> // return min(x(i))
T vecmin(Tp x, N n) {
  if (n == 0) return 0.0;
  T mn = x[0];
  for (N i = 1; i < n; i++) if (mn > x[i]) mn = x[i];
  return mn;
}

template<class T, class N, class Tp> // return min(x(i)) and the index
T vecmin(Tp x, N n, N &imin) {
  imin = 0;
  if (n == 0) return 0.0;
  T mn = x[0];
  for (N i = 1; i < n; i++) if (mn > x[i]) {mn = x[i]; imin = i;}
  return mn;
}

template<class T, class N, class Tp> // return max(x(i))
T vecmax(Tp x, N n) {
  if (n == 0) return 0.0;
  T mn = x[0];
  for (N i = 1; i < n; i++) if (mn < x[i]) mn = x[i];
  return mn;
}

template<class T, class N, class Tp> // return max(x(i)) and the index
T vecmax(Tp x, N n, N &imax) {
  imax = 0;
  if (n == 0) return 0.0;
  T mn = x[0];
  for (N i = 1; i < n; i++) if (mn < x[i]) {mn = x[i]; imax = i;}
  return mn;
}

template<class T> // x = 0
void zero(T x, Uint n) {
  for (Uint i = 0; i < n; i++) x[i] = 0;
}

template<class T, class Tp> // x = x/norm(x)
T normal(Tp x, Uint n) {
  T sum;
  Uint i;
  sum = 0;
  for (i=0; i<n; i++) sum = sum + x[i];
  if (sum != 0) {
    T factor = 1.0/sum;
    for (i=0; i<n; i++) x[i] = x[i]*factor;
  }
  return sum;
}

template<class T, class S> // d = s
void copyvec(T d, S s, Uint n) {
  for (Uint i = 0; i < n; i++) d[i] = s[i];
}

template<class T, class S> // d(i) = s
void copyval(T d, S s, Uint n) {
  for (Uint i = 0; i < n; i++) d[i] = s;
}

template<class T> // d(i) = a(i)*b(i)
void elemprod(T d, T a, T b, Uint n) {
  if (d == a)
    for (Uint i = 0; i < n; i++) d[i] *= b[i];
  else
    for (Uint i = 0; i < n; i++) d[i] = a[i]*b[i];
}

template<class T, class Tp> // return sum(x)
T sum(Tp x, Uint n) {
  T res = 0.0;
  for (Uint i = 0; i < n; i++) res += x[i];
  return res;
}

template<class T, class Tp>
void enlarge(Tp &s, Uint size, Uint incr) {
  // increase size of Double array s from size to size + incr
  Tp olds = s;
  NEWVEC(T, s, size + incr);
  copyvec(s, olds, size);
  DELETEVEC(olds);
}

template<class T>
void expandiv(T x, IV mask, Uint n) {
  if (mask != 0) {
    mask = ~mask;
    for (IV v = 0; v < n; v++)
      x[v] = x[v&mask];
  }
}

template<class Tp> // return x'*y
Double dotprod(Tp x, Tp y, Uint n) {
  Double sum = 0;
  for (Uint i = 0; i < n; i++) sum += x[i]*y[i];
  return sum;
}

template<class T, class S> // return d = a*x
void scalvec(T d, S a, T x, Uint n) {
  if (d == x) for (Uint i = 0; i < n; i++) x[i] *= a;
  else for (Uint i = 0; i < n; i++) d[i] = x[i]*a;
}

template<class T, class S> // d = a*x + y, may have d=x or d=y
void axpy(T d, S a, T x, T y, Uint n) {
  if (d == x) for (Uint i = 0; i < n; i++) { d[i] *= a; d[i] += y[i]; }
  else if (d == y) for (Uint i = 0; i < n; i++) d[i] += a*x[i];
  else for (Uint i = 0; i < n; i++) d[i] = a*x[i] + y[i];
}

template<class T, class S>
void mattvec(T y, S A, T x, Uint n, Uint m) { // y = A'*x, A nxm, may have y=x
  DoubleVec w;
  if (x == y) NEWVEC(Double, w, m); else w = y;
  for (Uint i = 0; i < m; i++) w[i] = dotprod(A[i], x, n);
  if (x == y) {copyvec(y, w, m); DELETEVEC(w);}
}

template<class T, class S> // y = A*x, A nxm, x m-vector, may have y=x
void matvec(T y, S A, T x, Uint n, Uint m) {
  DoubleVec w;
  if (x == y) NEWVEC(Double, w, n); 
  else w = y;
  zero(w, n);
  for (Uint i = 0; i < m; i++) axpy(w, x[i], A[i], y, n);
  if (x == y) {copyvec(y, w, n); DELETEVEC(w);}
}

// A := B*C (matrix multiply), may have A = C
template<class R, class S, class T> // B nxm, A and C have l rows
void matmat(R A, S B, T C, Uint n, Uint m, Uint l) {
  if (n == 0 || m == 0 || l == 0) return;
  if (A == C) {
    DoubleVec w;
    NEWVEC(Double, w, n);
    for (Uint i = 0; i < l; i++) {
      matvec(w, B, C[i], n, m);
      copyvec(A[i], w, n);
    }
    DELETEVEC(w);
  }
  else
    for (Uint i = 0; i < l; i++) matvec(A[i], B, C[i], n, m);
}

// A := B'*C (matrix multiply) may have A = C
template<class R, class S, class T> // B nxm, A and C have l rows
void mattmat(R A, S B, T C, Uint n, Uint m, Uint l) {
  if (n == 0 || m == 0 || l == 0) return;
  if (A == C) {
    DoubleVec w;
    NEWVEC(Double, w, m);
    for (Uint i = 0; i < l; i++) {
      mattvec(w, B, C[i], n, m);
      copyvec(A[i], w, m);
    }
    DELETEVEC(w);
  }
  else
    for (Uint i = 0; i < l; i++) mattvec(A[i], B, C[i], n, m);
}

template<class T>
Double det(T A, Uint n) {
  assertinternal(n < 3);
  if (n == 1) return A[0][0];
  else return A[0][0]*A[1][1] - A[0][1]*A[1][0];
}

#endif // _VECTORUTILS
